/* blm.c
 * converter for BLINKENmini movies
 *  - converts movies between different file formats
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "toolfunc.h"
#include "movie.h"
#include "blm.h"

//read a blm movie
struct tMovie * BlmRead( char * FileName )
{
	FILE * pFile;
	int LineNo, SizeX, SizeY, Size, I, PixelLine, FrameNo;
	char Buffer[4096], * pCommentNew;
	struct tMovie * pMovie;
	struct tFrame * pFrame;

	//open the file
	pFile = fopen( FileName, "rt" );
	if( pFile == NULL )
	{
		printf( "BlmRead: could not open \"%s\" for reading\n", FileName );
		return NULL;
	}
	LineNo = 0;

	//read first line
	LineNo++;
	if( ReadLine( pFile, Buffer, sizeof( Buffer ) ) == 1 )
		printf( "BlmRead: line %d too long in \"%s\" (truncated)\n", LineNo, FileName );
	//check identifier line
	if( strncmp( Buffer, "# BlinkenLights Movie ", 22 ) != 0 )
	{
		printf( "BlmRead: \"%s\" has invalid format (wrong identifier line)\n", FileName );
		fclose( pFile );
		return NULL;
	}
	//read size-x
	SizeX = 0;
	for( I = 22; Buffer[I] >= '0' && Buffer[I] <= '9'; I++ )
		SizeX = 10 * SizeX + Buffer[I] - '0';
	//check 'x'
	if( Buffer[I] != 'x' )
	{
		printf( "BlmRead: \"%s\" has invalid format (wrong size in identifier line)\n", FileName );
		fclose( pFile );
		return NULL;
	}
	I++; //jump over 'x'
	//read size-y
	SizeY = 0;
	for( ; Buffer[I] >= '0' && Buffer[I] <= '9'; I++ )
		SizeY = 10 * SizeY + Buffer[I] - '0';
	//check size
	if( SizeX == 0 || SizeY == 0 )
		printf( "BlmRead: \"%s\" has a strange frame size\n", FileName );

	//create a movie
	pMovie = MovieNew( );
	if( pMovie == NULL )
	{
		printf( "BlmRead: could not create a movie for \"%s\"\n", FileName );
		fclose( pFile );
		return NULL;
	}

	//put size into movie
	pMovie->SizeX = SizeX;
	pMovie->SizeY = SizeY;

	//read comment-lines
	while( ! feof( pFile ) )
	{
		//read next line
		LineNo++;
		if( ReadLine( pFile, Buffer, sizeof( Buffer ) ) == 1 )
			printf( "BlmRead: line %d too long in \"%s\" (truncated)\n", LineNo, FileName );
		//no empty line
		if( Buffer[0] != 0 )
		{
			//no comment-line
			if( Buffer[0] != '#' )
				//no more comments (comments may only be in front of first frame)
				break;
			//add comment line to comment buffer
			Size = strlen( pMovie->pComment ) + strlen( Buffer ) + 2;
			pCommentNew = (char *)realloc( pMovie->pComment, Size );
			if( pCommentNew == NULL )
			{
				printf( "BlmRead: could not resize comment memory block for \"%s\"\n", FileName );
				MovieFree( pMovie );
				fclose( pFile );
				return NULL;
			}
			pMovie->pComment = pCommentNew;
			strcat( pMovie->pComment, Buffer );
			strcat( pMovie->pComment, "\n" );
		}
	} //while( ! feof( pFile ) )

	//read frames
	pFrame = NULL; //no current frame yet
	FrameNo = 0;
	PixelLine = 0;
	for( ; ; )
	{
		//no empty line
		if( Buffer[0] != 0 )
		{
			//comment-line
			if( Buffer[0] == '#' )
				printf( "BlmRead: comment invalid in line %d of \"%s\" (ignored)\n", LineNo, FileName );
			//frame-start-line
			else if( Buffer[0] == '@' )
			{
				//put last frame to movie
				if( pFrame != NULL )
				{
					MovieAppendFrame( pMovie, pFrame );
					FrameNo++;
					pFrame = NULL; //no current frame
				}

				//create new frame
				pFrame = FrameNew( SizeX, SizeY );
				if( pFrame == NULL )
				{
					printf( "BlmRead: could not create frame in line %d of \"%s\"\n", LineNo, FileName );
					MovieFree( pMovie );
					fclose( pFile );
					return NULL;
				}
				//read duration
				pFrame->Duration = 0;
				for( I = 1; Buffer[I] >= '0' && Buffer[I] <= '9'; I++ )
					pFrame->Duration = pFrame->Duration * 10 + Buffer[I] - '0';
				//set pixels to off
				memset( pFrame->pPixel, 0, sizeof( unsigned char ) * SizeX * SizeY );
				//set current pixel line to first line
				PixelLine = 0;
			}
			//data-line
			else
			{
				//no current frame
				if( pFrame == NULL )
					printf( "BlmRead: data-line without current frame in line %d of \"%s\" (ignored)\n", LineNo, FileName );
				//current frame available
				else
				{
					//behind last line
					if( PixelLine >= SizeY )
						printf( "BlmRead: extra data-line behind frame in line %d of \"%s\" (ignored)\n", LineNo, FileName );
					//not behind last line
					else
					{
						//process data-line
						for( I = 0; I < SizeX && Buffer[I] != 0; I++ )
							if( Buffer[I] == '1' )
								pFrame->pPixel[PixelLine * SizeX + I] = 0xFF;
							else
								pFrame->pPixel[PixelLine * SizeX + I] = 0x00;
						//next line
						PixelLine++;
					}
				}
			}
		} //if( Buffer[0] != 0 )
		//file end reached
		if( feof( pFile ) )
			break;
		//read next line
		LineNo++;
		if( ReadLine( pFile, Buffer, sizeof( Buffer ) ) == 1 )
			printf( "BlmRead: line %d too long in \"%s\" (truncated)\n", LineNo, FileName );
	} //for( ; ; )

	//put last frame to movie
	if( pFrame != NULL )
	{
		MovieAppendFrame( pMovie, pFrame );
		FrameNo++;
		pFrame = NULL; //no current frame
	}

	//close the file
	fclose( pFile );

	//print info message
	printf( "BlmRead: file \"%s\" successfully read: %d frames\n", FileName, FrameNo );

	//return movie
	return pMovie;
}

//write a blm movie
int BlmWrite( struct tMovie * pMovie, char * FileName )
{
	FILE * pFile;
	struct tFrame * pFrame;
	unsigned int OmitX, InsertX, OmitY, InsertY, X, Y, FrameNo;

	//no movie
	if( pMovie == NULL )
		return -1;

	//open the file
	pFile = fopen( FileName, "wt" );
	if( pFile == NULL )
	{
		printf( "BlmWrite: could not open \"%s\" for writing\n", FileName );
		return -1;
	}

	//write the blm identification line
	fprintf( pFile, "# BlinkenLights Movie %dx%d\n", pMovie->SizeX, pMovie->SizeY );
	//write comments
	fprintf( pFile, "%s\n", pMovie->pComment );

	//write frames
	FrameNo = 0;
	for( pFrame = pMovie->pFirstFrame; pFrame != NULL; pFrame = pFrame->pNextFrame )
	{
		//write duration
		fprintf( pFile, "@%d\n", pFrame->Duration );

		//resize the frame to movie-global frame-size
		
		//calculate number of rows / colums to omit / to insert
		OmitX = 0;
		if( pMovie->SizeX < pFrame->SizeX )
			OmitX = pFrame->SizeX - pMovie->SizeX;
		InsertX = 0;
		if( pMovie->SizeX > pFrame->SizeX )
			InsertX = pMovie->SizeX - pFrame->SizeX;
		OmitY = 0;
		if( pMovie->SizeY < pFrame->SizeY )
			OmitY = pFrame->SizeY - pMovie->SizeY;
		InsertY = 0;
		if( pMovie->SizeY > pFrame->SizeY )
			InsertY = pMovie->SizeY - pFrame->SizeY;

		//write empty lines on top
		for( Y = 0; Y < InsertY / 2; Y++ )
		{
			for( X = 0; X < pMovie->SizeX; X++ )
				fprintf( pFile, "0" );
			fprintf( pFile, "\n" );
		}
		//write frame data
		for( Y = OmitY / 2; Y < pFrame->SizeY - (OmitY + 1) / 2; Y++ )
		{
			//write empty colums on the left
			for( X = 0; X < InsertX / 2; X++ )
				fprintf( pFile, "0" );
			//write row data
			for( X = OmitX / 2; X < pFrame->SizeX - (OmitX + 1) / 2; X++ )
				if( pFrame->pPixel[Y * pFrame->SizeX + X] >= 0x80 )
					fprintf( pFile, "1" );
				else
					fprintf( pFile, "0" );
			//write empty colums on the right
			for( X = 0; X < (InsertX + 1) / 2; X++ )
				fprintf( pFile, "0" );
			//write newline
			fprintf( pFile, "\n" );
		}
		//write empty lines on bottom
		for( Y = 0; Y < (InsertY + 1) / 2; Y++ )
		{
			for( X = 0; X < pMovie->SizeX; X++ )
				fprintf( pFile, "0" );
			fprintf( pFile, "\n" );
		}

		//write empty line behind frame
		fprintf( pFile, "\n" );

		FrameNo++;
	} //for( pFrame ...

	//close the file
	fclose( pFile );

	//print info message
	printf( "BlmWrite: file \"%s\" successfully written: %d frames\n", FileName, FrameNo );

	return 0;
}
