/* bmbm.c
 * converter for BLINKENmini movies
 *  - converts movies between different file formats
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>

#include "movie.h"
#include "bmbm.h"

//read a bmbm movie
struct tMovie * BmbmRead( char * FileName )
{
	FILE * pFile;
	int FrameNo;
	struct tMovie * pMovie;
	struct tFrame * pFrame;
	unsigned char SizeX, SizeY, Dur;

	//open the file
	pFile = fopen( FileName, "rb" );
	if( pFile == NULL )
	{
		printf( "BmbmRead: could not open \"%s\" for reading\n", FileName );
		return NULL;
	}

	//create a movie
	pMovie = MovieNew( );
	if( pMovie == NULL )
	{
		printf( "BmbmRead: could not create a movie for \"%s\"\n", FileName );
		fclose( pFile );
		return NULL;
	}

	//read frames
	pFrame = NULL;
	FrameNo = 0;
	for( ; ; )
	{
		//read size-x and size-y
		if( fread( &SizeX, sizeof( unsigned char ), 1, pFile ) != 1 )
			//could not read, so movie ends here
			break;
		if( fread( &SizeY, sizeof( unsigned char ), 1, pFile ) != 1 )
			//could not read, so movie ends here
			break;

		//create new frame
		pFrame = FrameNew( SizeX, SizeY );
		if( pFrame == NULL )
		{
			printf( "BmbmRead: could not create frame %d of \"%s\"\n", FrameNo, FileName );
			MovieFree( pMovie );
			fclose( pFile );
			return NULL;
		}

		//read pixels
		if( fread( pFrame->pPixel, sizeof( unsigned char ), SizeX * SizeY, pFile ) != (unsigned int)SizeX * (unsigned int)SizeY )
		{
			//could not read, so movie ends here
			FrameFree( pFrame );
			break;
		}

		//read duration
		if( fread( &Dur, sizeof( unsigned char ), 1, pFile ) != 1 )
		{
			//could not read, so movie ends here
			FrameFree( pFrame );
			break;
		}
		//write duration in milliseconds into frame
		pFrame->Duration = (unsigned int)Dur * 5;

		//append frame to movie
		MovieAppendFrame( pMovie, pFrame );
		FrameNo++;
		pFrame = NULL; //no current frame
	} //for( ; ; )

	//close the file
	fclose( pFile );

	//put maximum frame size into movie
	pMovie->SizeX = 0;
	pMovie->SizeY = 0;
	for( pFrame = pMovie->pFirstFrame; pFrame != NULL; pFrame = pFrame->pNextFrame )
	{
		if( pFrame->SizeX > pMovie->SizeX )
			pMovie->SizeX = pFrame->SizeX;
		if( pFrame->SizeY > pMovie->SizeY )
			pMovie->SizeY = pFrame->SizeY;
	}

	//print info message
	printf( "BmbmRead: file \"%s\" successfully read: %d frames\n", FileName, FrameNo );

	//return movie
	return pMovie;
}

//write a bmbm movie
int BmbmWrite( struct tMovie * pMovie, char * FileName )
{
	FILE * pFile;
	struct tFrame * pFrame;
	unsigned int OmitX, OmitY, Duration, Y, FrameNo;
	unsigned char SizeX, SizeY, Dur;

	//no movie
	if( pMovie == NULL )
		return -1;

	//open the file
	pFile = fopen( FileName, "wb" );
	if( pFile == NULL )
	{
		printf( "BmbmWrite: could not open \"%s\" for writing\n", FileName );
		return -1;
	}

	//write frames
	FrameNo = 0;
	for( pFrame = pMovie->pFirstFrame; pFrame != NULL; pFrame = pFrame->pNextFrame )
	{
		//calculate size and number of rows / colums to omit
		if( pFrame->SizeX <= 255 )
		{
			SizeX = (unsigned char)pFrame->SizeX;
			OmitX = 0;
		}
		else
		{
			SizeX = 255;
			OmitX = pFrame->SizeX - 255;
		}
		if( pFrame->SizeY <= 255 )
		{
			SizeY = (unsigned char)pFrame->SizeY;
			OmitY = 0;
		}
		else
		{
			SizeY = 255;
			OmitY = pFrame->SizeY - 255;
		}
		//calculate duration in 5ms-steps
		Duration = (pFrame->Duration + 2) / 5;

		//write frame as often as needed until duration is reached
		for( ; ; )
		{
			//write size-x and size-y as single byte
			fwrite( &SizeX, sizeof( unsigned char ), 1, pFile );
			fwrite( &SizeY, sizeof( unsigned char ), 1, pFile );
			//write lines
			for( Y = OmitY / 2; Y < pFrame->SizeY - (OmitY + 1) / 2; Y++ )
				//write line
				fwrite( pFrame->pPixel + Y * pFrame->SizeX + OmitX / 2, sizeof( unsigned char ), SizeX, pFile );
			//write duration
			if( Duration <= 255 )
				Dur = (unsigned char)Duration;
			else
				Dur = 255;
			fwrite( &Dur, sizeof( unsigned char ), 1, pFile );
			//calculate rest of duration
			Duration -= Dur;
			//done?
			if( Duration == 0 )
				break;
		}

		FrameNo++;
	} //for( pFrame ...

	//close the file
	fclose( pFile );

	//print info message
	printf( "BmbmWrite: file \"%s\" successfully written: %d frames\n", FileName, FrameNo );

	return 0;
}
