/* bmconv.c
 * converter for BLINKENmini movies
 *  - converts movies between different file formats
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>

#include "movie.h"
#include "blm.h"
#include "bmm.h"
#include "bmbm.h"
#include "gif.h"
#include "bml.h"

#define VerMaj 1
#define VerMin 0
#define VerRev 0

//main program
int main( int ArgC, char * * Args )
{
	struct tMovie * pMovie;
	int ArgErr, I;
	char InFmt, OutFmt; //input and output format: 'l'=blm, 'm'=bmm, 'b'=bmbm, 'g'=gif, 'x'=bml
	char * InFileName, * OutFileName;

	//print short message
	printf( "\n" );
	printf( "bmconv: converter for BLINKENmini movies (version %d.%d.%d)\n", VerMaj, VerMin, VerRev );
	printf( "        done in 2002 by sphaera & 1stein\n" );
	printf( "        http://blinkenmini.1stein.no-ip.com/\n" );
	printf( "        distributed under GNU Public License - no warranty\n\n" );

	//parse arguments
	ArgErr = 0;
	InFmt = 0;
	OutFmt = 0;
	InFileName = NULL;
	OutFileName = NULL;
	for( I = 1; I < ArgC; I++ )
	{
		//input is a blm
		if( strcmp( Args[I], "-l" ) == 0 || strcmp( Args[I], "--in-blm" ) == 0 )
			InFmt = 'l';
		//input is a bmm
		else if( strcmp( Args[I], "-m" ) == 0 || strcmp( Args[I], "--in-bbm" ) == 0 )
			InFmt = 'm';
		//input is a bmbm
		else if( strcmp( Args[I], "-b" ) == 0 || strcmp( Args[I], "--in-bmbm" ) == 0 )
			InFmt = 'b';
		//input is a gif
		else if( strcmp( Args[I], "-g" ) == 0 || strcmp( Args[I], "--in-gif" ) == 0 )
			InFmt = 'g';
		//input is a bml
		else if( strcmp( Args[I], "-x" ) == 0 || strcmp( Args[I], "--in-bml" ) == 0 )
			OutFmt = 'x';
		//output is a blm
		else if( strcmp( Args[I], "-L" ) == 0 || strcmp( Args[I], "--out-blm" ) == 0 )
			OutFmt = 'l';
		//output is a bmm
		else if( strcmp( Args[I], "-M" ) == 0 || strcmp( Args[I], "--out-bbm" ) == 0 )
			OutFmt = 'm';
		//output is a bmbm
		else if( strcmp( Args[I], "-B" ) == 0 || strcmp( Args[I], "--out-bmbm" ) == 0 )
			OutFmt = 'b';
		//output is a gif
		else if( strcmp( Args[I], "-G" ) == 0 || strcmp( Args[I], "--out-gif" ) == 0 )
			OutFmt = 'g';
		//output is a bml
		else if( strcmp( Args[I], "-X" ) == 0 || strcmp( Args[I], "--out-bml" ) == 0 )
			OutFmt = 'x';
		//input file
		else if( InFileName == NULL )
			InFileName = Args[I];
		//output file
		else if( OutFileName == NULL )
			OutFileName = Args[I];
		//argument error
		else
			ArgErr = 1;
	}

	//if no input-format try to guess from filename
	if( InFmt == 0 && InFileName != NULL )
	{
		I = strlen( InFileName );
		if( I >= 4 && strcmp( InFileName + I - 4, ".blm" ) == 0 )
			InFmt = 'l';
		if( I >= 4 && strcmp( InFileName + I - 4, ".bmm" ) == 0 )
			InFmt = 'm';
		if( I >= 5 && strcmp( InFileName + I - 5, ".bmbm" ) == 0 )
			InFmt = 'b';
		if( I >= 4 && strcmp( InFileName + I - 4, ".gif" ) == 0 )
			InFmt = 'g';
		if( I >= 4 && strcmp( InFileName + I - 4, ".bml" ) == 0 )
			InFmt = 'x';
	}
	//if no output-format try to guess from filename
	if( OutFmt == 0 && OutFileName != NULL )
	{
		I = strlen( OutFileName );
		if( I >= 4 && strcmp( OutFileName + I - 4, ".blm" ) == 0 )
			OutFmt = 'l';
		if( I >= 4 && strcmp( OutFileName + I - 4, ".bmm" ) == 0 )
			OutFmt = 'm';
		if( I >= 5 && strcmp( OutFileName + I - 5, ".bmbm" ) == 0 )
			OutFmt = 'b';
		if( I >= 4 && strcmp( OutFileName + I - 4, ".gif" ) == 0 )
			OutFmt = 'g';
		if( I >= 4 && strcmp( OutFileName + I - 4, ".bml" ) == 0 )
			OutFmt = 'x';
	}

	//filename(s) or format(s) missing
	if( InFileName == NULL || OutFileName == NULL || InFmt == 0 || OutFmt == 0 )
		ArgErr = 1;

	//print help
	if( ArgErr )
	{
		printf( "syntax: bmconv [<options>] <infile> <outfile>\n\n" );
		printf( "options: -l / -m / -b / -g / -x\n" );
		printf( "         --in-blm / --in-bmm / --in-bmbm / --in-gif / --in-bml\n" );
		printf( "         input file is a blm / bmm / bmbm / gif / bml\n\n" );
		printf( "         -L / -M / -B / -G / -X\n" );
		printf( "         --out-blm / --out-bmm / --out-bmbm / --out-gif / --out-bml\n" );
		printf( "         output file is a blm / bmm / bmbm / gif / bml\n\n" );
		printf( "if no format is given, it is guessed by the extension (.blm, .bmm, .bmbm, .gif, .bml)\n\n" );
		printf( "abbreviations: blm:  blinkenlights movie\n" );
		printf( "               bmm:  blinkenmini movie\n" );
		printf( "               bmbm: blinkenmini binary movie\n" );
		printf( "               gif:  graphics interchange format\n" );
		printf( "               bml:  blinkenlights markup language (XML like)\n\n" );
		return 1;
	}

	//print settings
	printf( "input file:    %s\n", InFileName );
	switch( InFmt )
	{
		case 'l': printf( "input format:  blm (blinkenlights movie)\n" ); break;
		case 'm': printf( "input format:  bmm (blinkenmini movie)\n" ); break;
		case 'b': printf( "input format:  bmbm (blinkenmini binary movie)\n" ); break;
		case 'g': printf( "input format:  gif (graphics interchange format)\n" ); break;
		case 'x': printf( "input format:  bml (blinkenlights markup language)\n" ); break;
	}
	printf( "output file:   %s\n", OutFileName );
	switch( OutFmt )
	{
		case 'l': printf( "output format: blm (blinkenlights movie)\n" ); break;
		case 'm': printf( "output format: bmm (blinkenmini movie)\n" ); break;
		case 'b': printf( "output format: bmbm (blinkenmini binary movie)\n" ); break;
		case 'g': printf( "output format: gif (graphics interchange format)\n" ); break;
		case 'x': printf( "output format: bml (blinkenlights markup language)\n" ); break;
	}
	printf( "\n" );

	//read movie
	switch( InFmt )
	{
		case 'l': pMovie = BlmRead( InFileName ); break;
		case 'm': pMovie = BmmRead( InFileName ); break;
		case 'b': pMovie = BmbmRead( InFileName ); break;
		case 'g': pMovie = GifRead( InFileName ); break;
		case 'x': pMovie = BmlRead( InFileName ); break;
		default: pMovie = NULL;
	}
	//error
	if( pMovie == NULL )
	{
		printf( "input error...\n\n" );
		return -1;
	}

	//write movie
	switch( OutFmt )
	{
		case 'l': I = BlmWrite( pMovie, OutFileName ); break;
		case 'm': I = BmmWrite( pMovie, OutFileName ); break;
		case 'b': I = BmbmWrite( pMovie, OutFileName ); break;
		case 'g': I = GifWrite( pMovie, OutFileName ); break;
		case 'x': I = BmlWrite( pMovie, OutFileName ); break;
		default: I = -1;
	}
	//error
	if( I != 0 )
	{
		printf( "output error...\n\n" );
		return -2;
	}

	//success
	printf( "done...\n\n" );
	return 0;
}

