/* gif.c
 * converter for BLINKENmini movies
 *  - converts movies between different file formats
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <gif_lib.h>

#include "movie.h"
#include "gif.h"

//read a movie from a gif-file
struct tMovie * GifRead( char * FileName )
{
	GifFileType * pGifFile;
	int FrameNo, I;
	struct tMovie * pMovie;
	struct tFrame * pFrame, * pLastFrame;
	SavedImage * pImage;
	ColorMapObject * pColorMap;
	unsigned char X, Y, ColorIndex, Color;
	unsigned short Duration, TranspColor;

	//open gif-file
	pGifFile = DGifOpenFileName( FileName );
	if( pGifFile == NULL )
	{
		printf( "GifRead: could not open \"%s\"\n", FileName );
		return NULL;
	}

	//read gif-images
	if( DGifSlurp( pGifFile ) == GIF_ERROR )
	{
		printf( "GifRead: could not read images in \"%s\"\n", FileName );
		DGifCloseFile( pGifFile );
		return NULL;
	}

	//create a movie
	pMovie = MovieNew( );
	if( pMovie == NULL )
	{
		printf( "GifRead: could not create a movie for \"%s\"\n", FileName );
		DGifCloseFile( pGifFile );
		return NULL;
	}
	//set size of movie
	pMovie->SizeX = pGifFile->SWidth;
	pMovie->SizeY = pGifFile->SHeight;

	//read frames
	pFrame = NULL;
	for( FrameNo = 0; FrameNo < pGifFile->ImageCount; FrameNo++ )
	{
		//remember last frame
		pLastFrame = pFrame;
		//create new frame
		pFrame = FrameNew( pMovie->SizeX, pMovie->SizeY );
		if( pFrame == NULL )
		{
			printf( "GifRead: could not create frame %d of \"%s\"\n", FrameNo, FileName );
			MovieFree( pMovie );
			DGifCloseFile( pGifFile );
			return NULL;
		}
		//copy last frame
		if( pLastFrame != NULL )
			memcpy( pFrame->pPixel, pLastFrame->pPixel, (unsigned int)pFrame->SizeX * (unsigned int)pFrame->SizeY );
		else
			memset( pFrame->pPixel, 0, (unsigned int)pFrame->SizeX * (unsigned int)pFrame->SizeY );

		//get gif-frame and color-map
		pImage = &pGifFile->SavedImages[FrameNo]; //get gif-frame
		pColorMap = pImage->ImageDesc.ColorMap; //get color-map
		if( pColorMap == NULL ) //no local-color-map
			pColorMap = pGifFile->SColorMap; //use global color-map
		if( pColorMap == NULL ) //no color-map at all
		{
			printf( "GifRead: no color map for frame %d of \"%s\"\n", FrameNo, FileName );
			FrameFree( pFrame );
			MovieFree( pMovie );
			DGifCloseFile( pGifFile );
			return NULL;
		}

		//get duration and transparent color
		Duration = 100; //default delay: a second
		TranspColor = 0x100; //impossible color
		for( I = 0; I < pImage->ExtensionBlockCount; I++ ) //search in all extensions of this frame
		{
			if( pImage->ExtensionBlocks[I].Function == GRAPHICS_EXT_FUNC_CODE ) //graphics-control extension
			{
				//get duration
				if( pImage->ExtensionBlocks[I].ByteCount >= 3 );
					Duration = *(unsigned short *)(pImage->ExtensionBlocks[I].Bytes + 1);
				//get transparent color
				if( pImage->ExtensionBlocks[I].ByteCount >= 4 && (pImage->ExtensionBlocks[I].Bytes[0] & 1) != 0 )
					TranspColor = (unsigned char)pImage->ExtensionBlocks[I].Bytes[3];
			}
		} //for( I ...
		//put duration into frame (in milliseconds)
		pFrame->Duration = Duration * 10;

		//read pixels from gif-frame
		for( Y = 0; Y < pImage->ImageDesc.Height; Y++ )
		{
			for( X = 0; X < pImage->ImageDesc.Width; X++ )
			{
				//get color-index of this pixel
				ColorIndex = pImage->RasterBits[Y * pImage->ImageDesc.Width + X];
				//get color of this index
				if( ColorIndex < pColorMap->ColorCount && ColorIndex != TranspColor )
					Color = (unsigned char)(((unsigned short)pColorMap->Colors[ColorIndex].Red
						      + (unsigned short)pColorMap->Colors[ColorIndex].Green
						      + (unsigned short)pColorMap->Colors[ColorIndex].Blue) / 3);
				else
					Color = 0;
				//write color into frame
				pFrame->pPixel[(Y + pImage->ImageDesc.Top) * pFrame->SizeX + X + pImage->ImageDesc.Left] = Color;
			} //for( X ...
		} //for( Y ...

		//append frame to movie
		MovieAppendFrame( pMovie, pFrame );
		pFrame = NULL; //no current frame
	} //for( FrameNo ...

	//close gif-file
	DGifCloseFile( pGifFile );

	//print info message
	printf( "GifRead: file \"%s\" successfully read: %d frames\n", FileName, FrameNo );

	//return movie
	return pMovie;
}

//write a movie to a gif-file
int GifWrite( struct tMovie * pMovie, char * FileName )
{
	GifFileType * pGifFile;
	struct tFrame * pFrame;
	int I, X, Y, FrameCnt, FrameNo, OmitX, OmitY, InsertX, InsertY, FrameX, FrameY;
	ColorMapObject * pColorMap;
	SavedImage * pSavedImages, * pImage;

	//no movie
	if( pMovie == NULL )
		return -1;

	//open the gif-file
	pGifFile = EGifOpenFileName( FileName, 0 );
	if( pGifFile == NULL )
	{
		printf( "GifWrite: could not open \"%s\" for writing\n", FileName );
		return -1;
	}

	//build color map
	pColorMap = (ColorMapObject *)malloc( sizeof( ColorMapObject ) ); //allocate color map object
	if( pColorMap == NULL )
	{
		printf( "GifWrite: could not allocate color map for \"%s\"\n", FileName );
		EGifCloseFile( pGifFile );
		return -1;
	}
	pColorMap->ColorCount = 256; //always use 256 colors
	pColorMap->BitsPerPixel = 8; //8 bits per pixel 
	pColorMap->Colors = (GifColorType *)malloc( pColorMap->ColorCount * sizeof( GifColorType ) ); //allocate buffer for colors
	if( pColorMap->Colors == NULL )
	{
		printf( "GifWrite: could not allocate color map for \"%s\"\n", FileName );
		free( pColorMap );
		EGifCloseFile( pGifFile );
		return -1;
	}
	for( I = 0; I < 256; I++ ) //write RGB-values of colors into buffer
	{
		//color with index I has RGB value I, I, I
		pColorMap->Colors[I].Red = I;
		pColorMap->Colors[I].Green = I;
		pColorMap->Colors[I].Blue = I;
	}

	//count frames
	FrameCnt = 0;
	for( pFrame = pMovie->pFirstFrame; pFrame != NULL; pFrame = pFrame->pNextFrame )
		FrameCnt++;
	//allocate buffer for saved images
	pSavedImages = (SavedImage *)malloc( FrameCnt * sizeof( SavedImage ) );
	if( pSavedImages == NULL )
	{
		printf( "GifWrite: could not allocate saved images structure for \"%s\"\n", FileName );
		free( pColorMap->Colors );
		free( pColorMap );
		EGifCloseFile( pGifFile );
		return -1;
	}

	//process frames
	FrameNo = 0;
	for( pFrame = pMovie->pFirstFrame; pFrame != NULL; pFrame = pFrame->pNextFrame )
	{
		//resize the frame to movie-global frame-size
		
		//calculate number of rows / colums to omit / to insert
		OmitX = 0;
		if( pMovie->SizeX < pFrame->SizeX )
			OmitX = pFrame->SizeX - pMovie->SizeX;
		InsertX = 0;
		if( pMovie->SizeX > pFrame->SizeX )
			InsertX = pMovie->SizeX - pFrame->SizeX;
		OmitY = 0;
		if( pMovie->SizeY < pFrame->SizeY )
			OmitY = pFrame->SizeY - pMovie->SizeY;
		InsertY = 0;
		if( pMovie->SizeY > pFrame->SizeY )
			InsertY = pMovie->SizeY - pFrame->SizeY;

		//get pointer to current image
		pImage = &pSavedImages[FrameNo];

		//fill image descriptor
		pImage->ImageDesc.Left = 0;
		pImage->ImageDesc.Top = 0;
		pImage->ImageDesc.Width = pMovie->SizeX;
		pImage->ImageDesc.Height = pMovie->SizeY;
		pImage->ImageDesc.Interlace = 0; //not interlaced
		pImage->ImageDesc.ColorMap = NULL; //use global color map

		//allocate buffer for raster bits
		pImage->RasterBits = (char *)malloc( pMovie->SizeX * pMovie->SizeY );
		if( pImage->RasterBits == NULL )
		{
			printf( "GifWrite: could not allocate pixel buffer for frame %d of \"%s\"\n", FrameNo, FileName );
			for( I = 0; I < FrameNo; I++ ) //free previos images
			{
				free( pSavedImages[I].ExtensionBlocks[0].Bytes );
				free( pSavedImages[I].ExtensionBlocks );
				free( pSavedImages[I].RasterBits );
			}
			free( pColorMap->Colors );
			free( pColorMap );
			EGifCloseFile( pGifFile );
			return -1;
		}
		//write raster bits
		for( Y = 0; Y < pMovie->SizeY; Y++ )
		{
			for( X = 0; X < pMovie->SizeX; X++ )
			{
				//get frame coordinates
				FrameX = X + OmitX / 2 - InsertX / 2;
				FrameY = Y + OmitY / 2 - InsertY / 2;
				//pixel in frame
				if( 0 <= FrameX && FrameX < (int)pFrame->SizeX && 0 <= FrameY && FrameY < (int)pFrame->SizeY )
					pImage->RasterBits[Y * pMovie->SizeX + X] = pFrame->pPixel[FrameY * pFrame->SizeX + FrameX];
				//pixel not in frame
				else
					pImage->RasterBits[Y * pMovie->SizeX + X] = 0;
			}
		}

		//no old-style function
		pImage->Function = 0;
		//one extension block with duration
		pImage->ExtensionBlockCount = 1;
		pImage->ExtensionBlocks = (ExtensionBlock *)malloc( sizeof( ExtensionBlock ) );
		if( pImage->ExtensionBlocks == NULL )
		{
			printf( "GifWrite: could not allocate extension block for frame %d of \"%s\"\n", FrameNo, FileName );
			free( pImage->RasterBits );
			for( I = 0; I < FrameNo; I++ ) //free previos images
			{
				free( pSavedImages[I].ExtensionBlocks[0].Bytes );
				free( pSavedImages[I].ExtensionBlocks );
				free( pSavedImages[I].RasterBits );
			}
			free( pColorMap->Colors );
			free( pColorMap );
			EGifCloseFile( pGifFile );
			return -1;
		}
		pImage->ExtensionBlocks[0].Function = GRAPHICS_EXT_FUNC_CODE; //function: graphis control information
		pImage->ExtensionBlocks[0].ByteCount = 4;
		pImage->ExtensionBlocks[0].Bytes = (char *)malloc( 4 );
		if( pImage->ExtensionBlocks[0].Bytes == NULL )
		{
			printf( "GifWrite: could not allocate extension block for frame %d of \"%s\"\n", FrameNo, FileName );
			free( pImage->ExtensionBlocks );
			free( pImage->RasterBits );
			for( I = 0; I < FrameNo; I++ ) //free previos images
			{
				free( pSavedImages[I].ExtensionBlocks[0].Bytes );
				free( pSavedImages[I].ExtensionBlocks );
				free( pSavedImages[I].RasterBits );
			}
			free( pColorMap->Colors );
			free( pColorMap );
			EGifCloseFile( pGifFile );
			return -1;
		}
		pImage->ExtensionBlocks[0].Bytes[0] = 0x00; //no transparent color
		pImage->ExtensionBlocks[0].Bytes[3] = 0x00; //the transparent color (ignored)
		*(unsigned short *)(pImage->ExtensionBlocks[0].Bytes + 1) = (unsigned short)((pFrame->Duration + 5) / 10); //the duration in 1/100 seconds

		FrameNo++;
	} //for( pFrame ...

	//put everything into the gif-structure
	pGifFile->SWidth = pMovie->SizeX; //size of image
	pGifFile->SHeight = pMovie->SizeY;
	pGifFile->SColorResolution = 1 << 24; //RGB colors
	pGifFile->SBackGroundColor = 0;
	pGifFile->SColorMap = pColorMap; //global color map
	pGifFile->ImageCount = FrameCnt; //number of images
	pGifFile->Image.Left = 0; //current image - no idea what this is good for - so everything 0 or NULL
	pGifFile->Image.Top = 0;
	pGifFile->Image.Width = 0;
	pGifFile->Image.Height = 0;
	pGifFile->Image.Interlace = 0;
	pGifFile->Image.ColorMap = NULL;
	pGifFile->SavedImages = pSavedImages; //the images

	//the color map belongs now to the gif file and is freed when closing the gif-file

	//write the gif
	if( EGifSpew( pGifFile ) == GIF_ERROR )
	{
		printf( "GifWrite: could not write \"%s\"\n", FileName );
		EGifCloseFile( pGifFile );
		return -1;
	}

	//gif-file is closed here (was done by EGifSpew)

	//free images
	for( I = 0; I < FrameCnt; I++ )
	{
		free( pSavedImages[I].ExtensionBlocks[0].Bytes );
		free( pSavedImages[I].ExtensionBlocks );
		free( pSavedImages[I].RasterBits ); 
	}

	//print info message
	printf( "GifWrite: file \"%s\" successfully written: %d frames\n", FileName, FrameNo );

	return 0;
}

