/* bml.c
 * converter for BLINKENmini movies
 *  - converts movies between different file formats
 * Copyright (C) 2002 sphaera & 1stein (blinkenmini@schuermans.info, https://blinkenmini.schuermans.info/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "movie.h"
#include "bml.h"

//read a bml movie
struct tMovie * BmlRead( char * FileName )
{
	FILE * pFile;
	int FileSize, SizeX, SizeY, Bits, SizeC, Size, Digits, I, FrameNo, Duration, PosX, PosY, Digit, C, Val;
	char * pData, *pDataStart, * pBlmStart, * pBlmEnd, * pHeaderStart, * pHeaderEnd, * pTagStart, * pTagEnd;
	char * pCommentNew, * pChr, * pFrameStart, * pFrameEnd;
	struct tMovie * pMovie;
	struct tFrame * pFrame;

	//open the file
	pFile = fopen( FileName, "rt" );
	if( pFile == NULL )
	{
		printf( "BmlRead: could not open \"%s\" for reading\n", FileName );
		return NULL;
	}

	//get file size
	if( fseek( pFile, 0, SEEK_END ) != 0 )
	{
		printf( "BmlRead: could not get size of \"%s\"\n", FileName );
		fclose( pFile );
		return NULL;
	}
	FileSize = ftell( pFile );
	if( FileSize < 0 )
	{
		printf( "BmlRead: could not get size of \"%s\"\n", FileName );
		fclose( pFile );
		return NULL;
	}
	if( fseek( pFile, 0, SEEK_SET ) != 0 )
	{
		printf( "BmlRead: could not get size of \"%s\"\n", FileName );
		fclose( pFile );
		return NULL;
	}

	//allocate buffer
	pData = (char *)malloc( FileSize + 1 );
	if( pData == NULL )
	{
		printf( "BmlRead: could not allocate buffer for data of \"%s\"\n", FileName );
		fclose( pFile );
		return NULL;
	}

	//read the file
	I = fread( pData, 1, FileSize, pFile );
	if( I < 0 )
	{
		printf( "BmlRead: could read \"%s\"\n", FileName );
		free( pData );
		fclose( pFile );
		return NULL;
	}
	pData[I] = 0;

	//close file
	fclose( pFile );

	//convert all CRs, LFs and TABs to spaces
	for( pChr = strchr( pData, '\r' ); pChr != NULL; pChr = strchr( pChr, '\r' ) )
		*pChr = ' ';
	for( pChr = strchr( pData, '\n' ); pChr != NULL; pChr = strchr( pChr, '\n' ) )
		*pChr = ' ';
	for( pChr = strchr( pData, '\t' ); pChr != NULL; pChr = strchr( pChr, '\t' ) )
		*pChr = ' ';

	//search blm-start-tag
	pBlmStart = strstr( pData, "<blm" );
	if( pBlmStart == NULL )
	{
		printf( "BmlRead: \"%s\" has invalid format: blm-tag not found\n", FileName );
		free( pData );
		return NULL;
	}
	//get movie size
	I = sscanf( pBlmStart, "<blm width=\"%d\" height=\"%d\" bits=\"%d\" channels=\"%d\">", &SizeX, &SizeY, &Bits, &SizeC );
	if( I != 4 )
	{
		SizeC = 1;
		I = sscanf( pBlmStart, "<blm width=\"%d\" height=\"%d\" bits=\"%d\">", &SizeX, &SizeY, &Bits );
		if( I != 3 )
		{
			printf( "BmlRead: \"%s\" has invaid format: blm-tag not correct\n", FileName );
			free( pData );
			return NULL;
		}
	}
	//check movie size
	if( SizeX < 0 )
		SizeX = -SizeX;
	if( SizeY < 0 )
		SizeY = -SizeY;
	if( Bits < 0 )
		Bits = -Bits;
	if( SizeC < 0 )
		SizeC = -SizeC;
	if( SizeX == 0 || SizeY == 0 )
		printf( "BmlRead: \"%s\" has a strange frame size\n", FileName );
	if( Bits == 0 || SizeC == 0 )
		printf( "BmlRead: \"%s\" has a strange color depth\n", FileName );
	//get number of digits per channel and pixel
	Digits = (Bits + 3) / 4;

	//search blm-end-tag
	pBlmEnd = strstr( pBlmStart, "</blm" );
	if( pBlmEnd == NULL )
	{
		printf( "BmlRead: \"%s\" has invalid format: blm-tag not closed\n", FileName );
		free( pData );
		return NULL;
	}
	//truncate data before blm-end-tag
	*pBlmEnd = 0;

	//data starts behind blm-start-tag
	pDataStart = pBlmStart + 1;

	//create a movie
	pMovie = MovieNew( );
	if( pMovie == NULL )
	{
		printf( "BmlRead: could not create a movie for \"%s\"\n", FileName );
		free( pData );
		return NULL;
	}

	//put size into movie
	pMovie->SizeC = SizeC;
	pMovie->SizeX = SizeX;
	pMovie->SizeY = SizeY;

	//search header-start-tag
	pHeaderStart = strstr( pBlmStart, "<header" );
	if( pHeaderStart != NULL )
	{
		//search header-end-tag
		pHeaderEnd = strstr( pHeaderStart, "</header" );
		if( pHeaderEnd == NULL )
		{
			printf( "BmlRead: \"%s\" has invalid format: header-tag not closed\n", FileName );
			MovieFree( pMovie );
			free( pData );
			return NULL;
		}
		//header ends at header-end-tag
		*pHeaderEnd = 0;
		//data starts behing header-end-tag
		pDataStart = pHeaderEnd + 1;

		//search title
		pTagStart = strstr( pHeaderStart, "<title>" );
		if( pTagStart != NULL )
		{
			//search end of title
			pTagEnd = strstr( pTagStart, "</title>" );
			if( pTagEnd == NULL )
			{
				printf( "BmlRead: \"%s\" has invalid format: title-tag not closed\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			//add title to comment
			Size = strlen( pMovie->pComment ) + (pTagEnd - pTagStart) + 64;
			pCommentNew = (char *)realloc( pMovie->pComment, Size );
			if( pCommentNew == NULL )
			{
				printf( "BmlRead: could not resize comment memory block for \"%s\"\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			pMovie->pComment = pCommentNew;
			strcat( pMovie->pComment, "# title: " );
			strncat( pMovie->pComment, pTagStart + 7, (pTagEnd - pTagStart - 7) );
			strcat( pMovie->pComment, "\n" );
		}

		//search description
		pTagStart = strstr( pHeaderStart, "<description>" );
		if( pTagStart != NULL )
		{
			//search end of description
			pTagEnd = strstr( pTagStart, "</description>" );
			if( pTagEnd == NULL )
			{
				printf( "BmlRead: \"%s\" has invalid format: description-tag not closed\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			//add description to comment
			Size = strlen( pMovie->pComment ) + (pTagEnd - pTagStart) + 64;
			pCommentNew = (char *)realloc( pMovie->pComment, Size );
			if( pCommentNew == NULL )
			{
				printf( "BmlRead: could not resize comment memory block for \"%s\"\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			pMovie->pComment = pCommentNew;
			strcat( pMovie->pComment, "# description: " );
			strncat( pMovie->pComment, pTagStart + 13, (pTagEnd - pTagStart - 13) );
			strcat( pMovie->pComment, "\n" );
		}

		//search author
		pTagStart = strstr( pHeaderStart, "<author>" );
		if( pTagStart != NULL )
		{
			//search end of author
			pTagEnd = strstr( pTagStart, "</author>" );
			if( pTagEnd == NULL )
			{
				printf( "BmlRead: \"%s\" has invalid format: author-tag not closed\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			//add author to comment
			Size = strlen( pMovie->pComment ) + (pTagEnd - pTagStart) + 64;
			pCommentNew = (char *)realloc( pMovie->pComment, Size );
			if( pCommentNew == NULL )
			{
				printf( "BmlRead: could not resize comment memory block for \"%s\"\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			pMovie->pComment = pCommentNew;
			strcat( pMovie->pComment, "# author: " );
			strncat( pMovie->pComment, pTagStart + 8, (pTagEnd - pTagStart - 8) );
			strcat( pMovie->pComment, "\n" );
		}

		//search email
		pTagStart = strstr( pHeaderStart, "<email>" );
		if( pTagStart != NULL )
		{
			//search end of email
			pTagEnd = strstr( pTagStart, "</email>" );
			if( pTagEnd == NULL )
			{
				printf( "BmlRead: \"%s\" has invalid format: email-tag not closed\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			//add email to comment
			Size = strlen( pMovie->pComment ) + (pTagEnd - pTagStart) + 64;
			pCommentNew = (char *)realloc( pMovie->pComment, Size );
			if( pCommentNew == NULL )
			{
				printf( "BmlRead: could not resize comment memory block for \"%s\"\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			pMovie->pComment = pCommentNew;
			strcat( pMovie->pComment, "# email: " );
			strncat( pMovie->pComment, pTagStart + 7, (pTagEnd - pTagStart - 7) );
			strcat( pMovie->pComment, "\n" );
		}

		//search url
		pTagStart = strstr( pHeaderStart, "<url>" );
		if( pTagStart != NULL )
		{
			//search end of url
			pTagEnd = strstr( pTagStart, "</url>" );
			if( pTagEnd == NULL )
			{
				printf( "BmlRead: \"%s\" has invalid format: url-tag not closed\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			//add url to comment
			Size = strlen( pMovie->pComment ) + (pTagEnd - pTagStart) + 64;
			pCommentNew = (char *)realloc( pMovie->pComment, Size );
			if( pCommentNew == NULL )
			{
				printf( "BmlRead: could not resize comment memory block for \"%s\"\n", FileName );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			pMovie->pComment = pCommentNew;
			strcat( pMovie->pComment, "# url: " );
			strncat( pMovie->pComment, pTagStart + 5, (pTagEnd - pTagStart - 5) );
			strcat( pMovie->pComment, "\n" );
		}

	} //if( pHeaderStart != NULL )


	//read frames
	FrameNo = 0;
	for( ; ; )
	{
		//seach frame-start
		pFrameStart = strstr( pDataStart, "<frame" );
		if( pFrameStart == NULL )
			break; //movie done
		//search header-end-tag
		pFrameEnd = strstr( pFrameStart, "</frame" );
		if( pFrameEnd == NULL )
		{
			printf( "BmlRead: \"%s\" has invalid format: frame-tag %d not closed\n", FileName, FrameNo );
			MovieFree( pMovie );
			free( pData );
			return NULL;
		}
		//frame ends after frame end-tag
		*pFrameEnd = 0;
		//next data starts behind this frame
		pDataStart = pFrameEnd + 1;

		//get duration
		I = sscanf( pFrameStart, "<frame duration=\"%d\">", &Duration );
		if( I != 1 )
		{
			printf( "BmlRead: \"%s\" has invaid format: frame-tag %d not correct\n", FileName, FrameNo );
			MovieFree( pMovie );
			free( pData );
			return NULL;
		}
		if( Duration < 0 )
			Duration = -Duration;

		//create new frame
		pFrame = FrameNew( SizeC, SizeX, SizeY );
		if( pFrame == NULL )
		{
			printf( "BmlRead: could not create frame %d of \"%s\"\n", FrameNo, FileName );
			MovieFree( pMovie );
			fclose( pFile );
			return NULL;
		}
		//set duration
		pFrame->Duration = Duration;
		//set pixels to off
		memset( pFrame->pPixel, 0, sizeof( unsigned char ) * SizeX * SizeY );

		//read rows
		for( PosY = 0; PosY < SizeY; PosY++ )
		{
			//search row-start-tag
			pTagStart = strstr( pFrameStart, "<row>" );
			if( pTagStart == NULL )
				break; //no more data for this frame
			//search row-end-tag
			pTagEnd = strstr( pTagStart, "</row>" );
			if( pTagEnd == NULL )
			{
				printf( "BmlRead: \"%s\" has invalid format: row-tag in frame %d not closed\n", FileName, FrameNo );
				FrameFree( pFrame );
				MovieFree( pMovie );
				free( pData );
				return NULL;
			}
			//next data starts behind this row
			pFrameStart = pTagEnd + 1;
			//process pixels in this row
			PosX = 0;
			Digit = 0;
			C = 0;
			Val = 0;
			for( I = 5; I < pTagEnd - pTagStart; I++ )
			{
				//read digit
				Val *= 0x10;
				if( pTagStart[I] >= '0' && pTagStart[I] <= '9' )
					Val += pTagStart[I] - '0';
				if( pTagStart[I] >= 'A' && pTagStart[I] <= 'F' )
					Val += pTagStart[I] + 0xA - 'A';
				if( pTagStart[I] >= 'a' && pTagStart[I] <= 'f' )
					Val += pTagStart[I] + 0xA - 'a';
				//next digit
				Digit++;
				if( Digit >= Digits )
				{
					//store value in pixel
					if( Bits < 8 )
						Val <<= 8 - Bits;
					if( Bits > 8 )
						Val >>= Bits - 8;
					pFrame->pPixel[(PosY * SizeX + PosX) * SizeC + C] = (unsigned char)Val;
					//next channel
					Digit = 0;
					C++;
					if( C >= SizeC )
					{
						//next pixel
						C = 0;
						PosX++;
						//end of line reached
						if( PosX >= SizeX )
							break;
					}
				}
			}
		} //for( PosY ...

		//put frame into movie
		MovieAppendFrame( pMovie, pFrame );
		FrameNo++;

	} //for( ; ; )

	//print info message
	printf( "BmlRead: file \"%s\" successfully read: %d frames\n", FileName, FrameNo );

	//return movie
	return pMovie;
}

//write a bml movie
int BmlWrite( struct tMovie * pMovie, char * FileName )
{
	FILE * pFile;
	char * pComment, * pChr;
	struct tFrame * pFrame;
	unsigned int ReduceC, OmitX, InsertX, OmitY, InsertY, CM, CF, CR, X, Y, FrameNo, Val;

	//no movie
	if( pMovie == NULL )
		return -1;

	//open the file
	pFile = fopen( FileName, "wt" );
	if( pFile == NULL )
	{
		printf( "BmlWrite: could not open \"%s\" for writing\n", FileName );
		return -1;
	}

	//write xml-line
	fprintf( pFile, "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n" );
	//write blm-tag
	fprintf( pFile, "<blm width=\"%d\" height=\"%d\" bits=\"8\" channels=\"%d\">\n", pMovie->SizeX, pMovie->SizeY, pMovie->SizeC );

	//write header
	fprintf( pFile, "    <header>\n" );
	fprintf( pFile, "        <title>%s</title>\n", FileName ); //write filename as title
	pComment = pMovie->pComment; //copy comment
	for( pChr = strchr( pComment, '\r' ); pChr != NULL; pChr = strchr( pChr, '\r' ) ) //remove CRs and LFs from comment
		*pChr = ' ';
	for( pChr = strchr( pComment, '\n' ); pChr != NULL; pChr = strchr( pChr, '\n' ) )
		*pChr = ' ';
	fprintf( pFile, "        <description>%s</description>\n", pComment ); //write comment as description
	free( pComment ); //free copy of comment
	fprintf( pFile, "        <creator>bmconv</creator>\n" );
	fprintf( pFile, "    </header>\n" );

	//write frames
	FrameNo = 0;
	for( pFrame = pMovie->pFirstFrame; pFrame != NULL; pFrame = pFrame->pNextFrame )
	{
		//write frame-tag
		fprintf( pFile, "\n    <frame duration=\"%d\">\n", pFrame->Duration );

		//resize the frame to movie-global frame-size
		
		//calculate number of channels to reduce to one
		ReduceC = (pFrame->SizeC - 1) / pMovie->SizeC + 1;
		//calculate number of rows / colums to omit / to insert
		OmitX = 0;
		if( pMovie->SizeX < pFrame->SizeX )
			OmitX = pFrame->SizeX - pMovie->SizeX;
		InsertX = 0;
		if( pMovie->SizeX > pFrame->SizeX )
			InsertX = pMovie->SizeX - pFrame->SizeX;
		OmitY = 0;
		if( pMovie->SizeY < pFrame->SizeY )
			OmitY = pFrame->SizeY - pMovie->SizeY;
		InsertY = 0;
		if( pMovie->SizeY > pFrame->SizeY )
			InsertY = pMovie->SizeY - pFrame->SizeY;

		//write empty lines on top
		for( Y = 0; Y < InsertY / 2; Y++ )
		{
			fprintf( pFile, "        <row>" );
			for( X = 0; X < pMovie->SizeX; X++ )
				fprintf( pFile, "00" );
			fprintf( pFile, "</row>\n" );
		}
		//write frame data
		for( Y = OmitY / 2; Y < pFrame->SizeY - (OmitY + 1) / 2; Y++ )
		{
			//write row-tag
			fprintf( pFile, "        <row>" );
			//write empty colums on the left
			for( X = 0; X < InsertX / 2; X++ )
				fprintf( pFile, "00" );
			//write row data
			for( X = OmitX / 2; X < pFrame->SizeX - (OmitX + 1) / 2; X++ )
			{
				//write pixel data
				for( CM = 0, CF = 0; CM < pMovie->SizeC; CM++ )
				{
					//reduce ReduceC channels to one
					Val = 0;
					for( CR = 0; CR < ReduceC && CF < pFrame->SizeC; CR++, CF++ )
						Val += pFrame->pPixel[(Y * pFrame->SizeX + X) * pFrame->SizeC + CF];
					if( ReduceC > 0 )
						Val /= ReduceC;
					//write channel value
					fprintf( pFile, "%02X", (unsigned char)Val );
				}
			}
			//write empty colums on the right
			for( X = 0; X < (InsertX + 1) / 2; X++ )
				fprintf( pFile, "00" );
			//write row-end-tag
			fprintf( pFile, "</row>\n" );
		}
		//write empty lines on bottom
		for( Y = 0; Y < (InsertY + 1) / 2; Y++ )
		{
			fprintf( pFile, "        <row>" );
			for( X = 0; X < pMovie->SizeX; X++ )
				fprintf( pFile, "00" );
			fprintf( pFile, "</row>\n" );
		}

		//write frame-end-tag
		fprintf( pFile, "    </frame>\n" );

		FrameNo++;
	} //for( pFrame ...

	//write blm-end-tag
	fprintf( pFile, "</blm>\n" );

	//close the file
	fclose( pFile );

	//print info message
	printf( "BmlWrite: file \"%s\" successfully written: %d frames\n", FileName, FrameNo );

	return 0;
}

