/* bmm.c
 * converter for BLINKENmini movies
 *  - converts movies between different file formats
 * Copyright (C) 2002 sphaera & 1stein (blinkenmini@schuermans.info, https://blinkenmini.schuermans.info/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "toolfunc.h"
#include "movie.h"
#include "bmm.h"

//read a bmm movie
struct tMovie * BmmRead( char * FileName )
{
	FILE * pFile;
	int LineNo, SizeX, SizeY, Size, I, PixelLine, FrameNo;
	char Buffer[4096], * pCommentNew, * pChr, * pChrSpace, * pChrTab;
	struct tMovie * pMovie;
	struct tFrame * pFrame;
	unsigned char Val;

	//open the file
	pFile = fopen( FileName, "rt" );
	if( pFile == NULL )
	{
		printf( "BmmRead: could not open \"%s\" for reading\n", FileName );
		return NULL;
	}
	LineNo = 0;

	//read first line
	LineNo++;
	if( ReadLine( pFile, Buffer, sizeof( Buffer ) ) == 1 )
		printf( "BmmRead: line %d too long in \"%s\" (truncated)\n", LineNo, FileName );
	//check identifier line
	if( strncmp( Buffer, "# BlinkenMini Movie ", 20 ) != 0 )
	{
		printf( "BmmRead: \"%s\" has invalid format (wrong identifier line)\n", FileName );
		fclose( pFile );
		return NULL;
	}
	//read size-x
	SizeX = 0;
	for( I = 20; Buffer[I] >= '0' && Buffer[I] <= '9'; I++ )
		SizeX = 10 * SizeX + Buffer[I] - '0';
	//check 'x'
	if( Buffer[I] != 'x' )
	{
		printf( "BmmRead: \"%s\" has invalid format (wrong size in identifier line)\n", FileName );
		fclose( pFile );
		return NULL;
	}
	I++; //jump over 'x'
	//read size-y
	SizeY = 0;
	for( ; Buffer[I] >= '0' && Buffer[I] <= '9'; I++ )
		SizeY = 10 * SizeY + Buffer[I] - '0';
	//check size
	if( SizeX == 0 || SizeY == 0 )
		printf( "BmmRead: \"%s\" has a strange frame size\n", FileName );

	//create a movie
	pMovie = MovieNew( );
	if( pMovie == NULL )
	{
		printf( "BmmRead: could not create a movie for \"%s\"\n", FileName );
		fclose( pFile );
		return NULL;
	}

	//put size into movie
	pMovie->SizeC = 1;
	pMovie->SizeX = SizeX;
	pMovie->SizeY = SizeY;

	//read comment-lines
	while( ! feof( pFile ) )
	{
		//read next line
		LineNo++;
		if( ReadLine( pFile, Buffer, sizeof( Buffer ) ) == 1 )
			printf( "BmmRead: line %d too long in \"%s\" (truncated)\n", LineNo, FileName );
		//no empty line
		if( Buffer[0] != 0 )
		{
			//no comment-line
			if( Buffer[0] != '#' )
				//no more comments (comments may only be in front of first frame)
				break;
			//add comment line to comment buffer
			Size = strlen( pMovie->pComment ) + strlen( Buffer ) + 2;
			pCommentNew = (char *)realloc( pMovie->pComment, Size );
			if( pCommentNew == NULL )
			{
				printf( "BmmRead: could not resize comment memory block for \"%s\"\n", FileName );
				MovieFree( pMovie );
				fclose( pFile );
				return NULL;
			}
			pMovie->pComment = pCommentNew;
			strcat( pMovie->pComment, Buffer );
			strcat( pMovie->pComment, "\n" );
		}
	} //while( ! feof( pFile ) )

	//read frames
	pFrame = NULL; //no current frame yet
	FrameNo = 0;
	PixelLine = 0;
	for( ; ; )
	{
		//no empty line
		if( Buffer[0] != 0 )
		{
			//comment-line
			if( Buffer[0] == '#' )
				printf( "BmmRead: comment invalid in line %d of \"%s\" (ignored)\n", LineNo, FileName );
			//frame-start-line
			else if( Buffer[0] == '@' )
			{
				//put last frame to movie
				if( pFrame != NULL )
				{
					MovieAppendFrame( pMovie, pFrame );
					FrameNo++;
					pFrame = NULL; //no current frame
				}

				//create new frame
				pFrame = FrameNew( 1, SizeX, SizeY );
				if( pFrame == NULL )
				{
					printf( "BmmRead: could not create frame in line %d of \"%s\"\n", LineNo, FileName );
					MovieFree( pMovie );
					fclose( pFile );
					return NULL;
				}
				//read duration
				pFrame->Duration = 0;
				for( I = 1; Buffer[I] >= '0' && Buffer[I] <= '9'; I++ )
					pFrame->Duration = pFrame->Duration * 10 + Buffer[I] - '0';
				//set pixels to off
				memset( pFrame->pPixel, 0, sizeof( unsigned char ) * SizeX * SizeY );
				//set current pixel line to first line
				PixelLine = 0;
			}
			//data-line
			else
			{
				//no current frame
				if( pFrame == NULL )
					printf( "BmmRead: data-line without current frame in line %d of \"%s\" (ignored)\n", LineNo, FileName );
				//current frame available
				else
				{
					//behind last line
					if( PixelLine >= SizeY )
						printf( "BmmRead: extra data-line behind frame in line %d of \"%s\" (ignored)\n", LineNo, FileName );
					//not behind last line
					else
					{
						//process data-line
						pChr = Buffer; //start at beginning of line
						for( I = 0; I < SizeX; I++ )
						{
							//read value
							Val = 0;
							if( *pChr == '0' ) //first character is '0'
							{
								pChr++;
								if( *pChr >= '0' && *pChr <= '7' ) //second character is '0'..'9'
								{
									//read octal number
									for( ; *pChr >= '0' && *pChr <= '7'; pChr++ )
										Val = Val * 8 + *pChr - '0';
								}
								else if( *pChr == 'X' || *pChr == 'x' ) //second character is 'X' or 'x'
								{
									pChr++;
									//read hexadecimal number
									for( ; ; pChr++ )
										if( *pChr >= '0' && *pChr <= '9' )
											Val = Val * 16 + *pChr - '0';
										else if( *pChr >= 'A' && *pChr <= 'F' )
											Val = Val * 16 + *pChr + 10 - 'A';
										else if( *pChr >= 'a' && *pChr <= 'f' )
											Val = Val * 16 + *pChr + 10 - 'a';
										else
											break;
								}
							}
							else if( *pChr >= '1' && *pChr <= '9' ) //fist character is '1'..'9'
							{
								//read decimal number
								for( ; *pChr >= '0' && *pChr <= '9'; pChr++ )
									Val = Val * 10 + *pChr - '0';
							}
							//save value in pixel
							pFrame->pPixel[PixelLine * SizeX + I] = Val;
							//go to next value in line
							pChrSpace = strchr( pChr, ' ' ); //search next space or tab
							pChrTab = strchr( pChr, '\t' );
							if( pChrSpace == NULL )
								if( pChrTab == NULL )
									break; //no space, no tab, so end loop
								else
									pChr = pChrTab + 1; //no space, tab there, so go behind tab
							else
								if( pChrTab == NULL )
									pChr = pChrSpace + 1; //space there, no tab, so got behind space
								else
									if( pChrSpace < pChrTab ) //space there, tab there, so go behind first one
										pChr = pChrSpace + 1;
									else
										pChr = pChrTab + 1;
						}
						//next line
						PixelLine++;
					}
				}
			}
		} //if( Buffer[0] != 0 )
		//file end reached
		if( feof( pFile ) )
			break;
		//read next line
		LineNo++;
		if( ReadLine( pFile, Buffer, sizeof( Buffer ) ) == 1 )
			printf( "BmmRead: line %d too long in \"%s\" (truncated)\n", LineNo, FileName );
	} //for( ; ; )

	//put last frame to movie
	if( pFrame != NULL )
	{
		MovieAppendFrame( pMovie, pFrame );
		FrameNo++;
		pFrame = NULL; //no current frame
	}

	//close the file
	fclose( pFile );

	//print info message
	printf( "BmmRead: file \"%s\" successfully read: %d frames\n", FileName, FrameNo );

	//return movie
	return pMovie;
}

//write a bmm movie
int BmmWrite( struct tMovie * pMovie, char * FileName )
{
	FILE * pFile;
	struct tFrame * pFrame;
	unsigned int OmitX, InsertX, OmitY, InsertY, X, Y, FrameNo, Val, C;

	//no movie
	if( pMovie == NULL )
		return -1;

	//open the file
	pFile = fopen( FileName, "wt" );
	if( pFile == NULL )
	{
		printf( "BmmWrite: could not open \"%s\" for writing\n", FileName );
		return -1;
	}

	//write the bmm identification line
	fprintf( pFile, "# BlinkenMini Movie %dx%d\n", pMovie->SizeX, pMovie->SizeY );
	//write comments
	fprintf( pFile, "%s\n", pMovie->pComment );

	//write frames
	FrameNo = 0;
	for( pFrame = pMovie->pFirstFrame; pFrame != NULL; pFrame = pFrame->pNextFrame )
	{
		//write duration
		fprintf( pFile, "@%d\n", pFrame->Duration );

		//resize the frame to movie-global frame-size
		
		//calculate number of rows / colums to omit / to insert
		OmitX = 0;
		if( pMovie->SizeX < pFrame->SizeX )
			OmitX = pFrame->SizeX - pMovie->SizeX;
		InsertX = 0;
		if( pMovie->SizeX > pFrame->SizeX )
			InsertX = pMovie->SizeX - pFrame->SizeX;
		OmitY = 0;
		if( pMovie->SizeY < pFrame->SizeY )
			OmitY = pFrame->SizeY - pMovie->SizeY;
		InsertY = 0;
		if( pMovie->SizeY > pFrame->SizeY )
			InsertY = pMovie->SizeY - pFrame->SizeY;

		//write empty lines on top
		for( Y = 0; Y < InsertY / 2; Y++ )
		{
			for( X = 0; X < pMovie->SizeX; X++ )
				fprintf( pFile, "0x00 " );
			fprintf( pFile, "\n" );
		}
		//write frame data
		for( Y = OmitY / 2; Y < pFrame->SizeY - (OmitY + 1) / 2; Y++ )
		{
			//write empty colums on the left
			for( X = 0; X < InsertX / 2; X++ )
				fprintf( pFile, "0x00 " );
			//write row data
			for( X = OmitX / 2; X < pFrame->SizeX - (OmitX + 1) / 2; X++ )
			{
				//calculate average of all channles
				Val = 0;
				for( C = 0; C < pFrame->SizeC; C++ )
					Val += pFrame->pPixel[(Y * pFrame->SizeX + X) * pFrame->SizeC + C];
				if( pFrame->SizeC > 0 )
					Val /= pFrame->SizeC;
				//write average value
				fprintf( pFile, "0x%02X ", (unsigned char)Val );
			}
			//write empty colums on the right
			for( X = 0; X < (InsertX + 1) / 2; X++ )
				fprintf( pFile, "0x00 " );
			//write newline
			fprintf( pFile, "\n" );
		}
		//write empty lines on bottom
		for( Y = 0; Y < (InsertY + 1) / 2; Y++ )
		{
			for( X = 0; X < pMovie->SizeX; X++ )
				fprintf( pFile, "0x00 " );
			fprintf( pFile, "\n" );
		}

		//write empty line behind frame
		fprintf( pFile, "\n" );

		FrameNo++;
	} //for( pFrame ...

	//close the file
	fclose( pFile );

	//print info message
	printf( "BmmWrite: file \"%s\" successfully written: %d frames\n", FileName, FrameNo );

	return 0;
}
