/* bmdrv.c
 * kernel driver for BLINKENmini
 *  - character device kernel module for input from and output to BLINKENmini using the parallel port
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/ioport.h>
#include <linux/fs.h>
#include <linux/wrapper.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/param.h>
#include <linux/interrupt.h>

//number of elements in an array
#define count( array ) (sizeof( (array) ) / sizeof( (array)[0] ))

//version information for insmod
char version[] = UTS_RELEASE;

//startup parameters
int io = 0;
MODULE_PARM( io, "i" );
int irq = 0;
MODULE_PARM( irq, "i" );

//include bmdrv-header-file
#include "bmdrv.h"

//global constants concerning the module
#define BmDrvDevName "bmdrv" //the name of this device (max. 31 chars allowed)
#define BmDrvDevId ((void *)(('b'<<24)+('m'<<16)+('d'<<8)+'r')) //the id of this device
#define BmDrvDevVerMaj 1 //version of this device
#define BmDrvDevVerMin 1
#define BmDrvDevVerRev 0

//global types
#define BmDrvInternalPictureSizeX 18 //dimensions of an internal BM-picture
#define BmDrvInternalPictureSizeY 8
typedef struct tBmDrvInternalPicture //an internal BM-picture
{
	unsigned char Pixels[BmDrvInternalPictureSizeY][BmDrvInternalPictureSizeX];
} stBmDrvInternalPicture;
#define BmDrvInternalButtonCnt 16 //number of buttons per controller
#define BmDrvInternalControllerCnt 4 //number of controllers
#define BmDrvInternalButtonHyst 16 //width of hysteresis
typedef struct tBmDrvInternalControllerButtons //state of the internal controller buttons
{
	unsigned char Buttons[BmDrvInternalControllerCnt][BmDrvInternalButtonCnt];
} stBmDrvInternalControllerButtons;

//global variables
int BmDrvOutIsOpen = 0; //true if the output device is opened by a process
int BmDrvInIsOpen = 0; //true if the input device is opened by a process
struct file * BmDrvOutFile; //file-struct of output-file (NULL if closed)
struct file * BmDrvInFile; //file-struct of input-file (NULL if closed)
unsigned short BmDrvIoAddrs[] = { 0x3BC, 0x378, 0x278 }; //valid io-addresses
unsigned short BmDrvIoAddr; //io-address
unsigned char BmDrvIrqNos[] = { 7, 5 }; //valid irq-numbers
unsigned char BmDrvIrqNo; //irq-number

//brightness-level to duty-cycle conversion
#define BmDrvOutputSteps 18
unsigned char BmDrvOutputStepPixelVal[BmDrvOutputSteps] =
{ 0x20, 0x40, 0x60, 0x78, 0x90, 0xA8, 0xB8, 0xC8, 0xD8, 0xDC, 0xE0, 0xE4, 0xE8, 0xEC, 0xF4, 0xF8, 0xFC, 0xFE };
//the next step of the output-process to take when an interrupt occurs
int BmDrvOutputStep = 0;

//the current picture that is being displayed
struct tBmDrvInternalPicture BmDrvCurrentPicture;

//the next picture to display
//this picture exists two times
//one can be altered by the ioctl-syscall
//the other one can be read by the interrupt (which can happen at any time)
struct tBmDrvInternalPicture BmDrvNextPictures[2];
//the index of the picture that currently belongs to the interrupt
int BmDrvNextPictureInterruptIndex;

/*
//the standard-picture: diagonal lines
struct tBmDrvInternalPicture BmDrvStdPicture =
{
Pixels:
	{
		{ 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, },
		{ 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, },
		{ 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, },
		{ 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, },
		{ 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, },
		{ 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, },
		{ 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, },
		{ 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, },
 	}
};
*/

/*
//the standard-picture: BM
struct tBmDrvInternalPicture BmDrvStdPicture =
{
Pixels:
	{
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
 	}
};
*/

//the standard-picture: BM with shadow
struct tBmDrvInternalPicture BmDrvStdPicture =
{
Pixels:
	{
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x40, 0x40, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x40, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0x40, 0x40, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0x40, 0x40, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x40, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x00, 0x00, 0x00, 0x00, 0x40, 0x40, },
 	}
};

//hysteresis-counters for the buttons (used only in interrupt)
struct tBmDrvInternalControllerButtons BmDrvButtonHyst;
//last button state (used only in interrupt)
struct tBmDrvInternalControllerButtons BmDrvLastButtonState;
//current button state
struct tBmDrvInternalControllerButtons BmDrvButtonState;
//change-flag for button state
//(set by interrupt)
//can be used to determine if it is neccessary to re-read button state
int BmDrvButtonStateChanged;

//global variables for writing to the device in bmbm-format
int BmDrvWriteSizeX, BmDrvWriteSizeY; //the size of the frame being being written (-1: size not written yet)
int BmDrvWriteVisibleL, BmDrvWriteVisibleR, BmDrvWriteVisibleT, BmDrvWriteVisibleB; //left, right, top, bottom of visible area of picture
int BmDrvWriteOffsetX, BmDrvWriteOffsetY; //offset of topleft-pixel of written picture from topleft-pixel of display
int BmDrvWritePosX, BmDrvWritePosY; //position of the next pixel that has to be written
struct tBmDrvInternalPicture BmDrvWritePicture; //the picture with the data already written
int BmDrvWriteDelay; //the delay of the frame currently being written (-1: delay not written yet)

//global variables for reading from the device
//format is 4 bytes per event:
//  0xFF <action> <controller-no> <button-no>
//  action: 0x00: button released, 0x01: button pressed
//  example: 0xFF 0x01 0x00 0x02 ---> button 2 of controller 0 was pressed
#define BmDrvReadBufferSize 64
unsigned char BmDrvReadBuffer[BmDrvReadBufferSize][4]; //ring-buffer with actions
int BmDrvReadBufferStart; //start-index for writing into buffer
int BmDrvReadBufferEnd; //end-index for writing into buffer
unsigned char BmDrvReadChrBuffer[4]; //buffer for left characters from the last read
int BmDrvReadChrBufferCnt; //number of characters in buffer for left characters from the last read

//write an event into the read buffer
int BmDrvReadBufferWrite( unsigned char Event0, unsigned char Event1, unsigned char Event2, unsigned char Event3 )
{
	//local variables
	int NewEnd;

	//calculate new end-index
	NewEnd = BmDrvReadBufferEnd + 1;
	if( NewEnd >= BmDrvReadBufferSize )
		NewEnd = 0;

	//new end-index equal to start-index
	//buffer is full, ignore this action
	if( NewEnd == BmDrvReadBufferStart )
		//error
		return -1;

	//write event into read buffer
	BmDrvReadBuffer[BmDrvReadBufferEnd][0] = Event0;
	BmDrvReadBuffer[BmDrvReadBufferEnd][1] = Event1;
	BmDrvReadBuffer[BmDrvReadBufferEnd][2] = Event2;
	BmDrvReadBuffer[BmDrvReadBufferEnd][3] = Event3;

	//set new end-index
	BmDrvReadBufferEnd = NewEnd;

	//success
	return 0;
}

//read an event from the read buffer
int BmDrvReadBufferRead( unsigned char Event[4] )
{
	//local variables
	int NewStart;

	//start-index equal to end-index
	//buffer is empty, nothing to read
	if( BmDrvReadBufferStart == BmDrvReadBufferEnd )
		//error
		return -1;

	//return event to caller
	Event[0] = BmDrvReadBuffer[BmDrvReadBufferStart][0];
	Event[1] = BmDrvReadBuffer[BmDrvReadBufferStart][1];
	Event[2] = BmDrvReadBuffer[BmDrvReadBufferStart][2];
	Event[3] = BmDrvReadBuffer[BmDrvReadBufferStart][3];

	//calculate new start-index
	NewStart = BmDrvReadBufferStart + 1;
	if( NewStart >= BmDrvReadBufferSize )
		NewStart = 0;

	//set new start-index
	BmDrvReadBufferStart = NewStart;

	//success
	return 0;
}

//output a frame
//turns on all pixels of *pPicture that have a value >= PixelValue
//if pPicture is NULL, all pixels are turned off
void BmDrvFrameOut( struct tBmDrvInternalPicture * pPicture, unsigned char PixelValue )
{
	//local variables
	int X, Y;
	unsigned char Val;

	//loop through all columns
	//(backwards: leftmost pixel as last one)
	for( X = BmDrvInternalPictureSizeX - 1; X >= 0; X-- )
	{
		//calculate value for this column
		Val = 0;
		if( pPicture != NULL )
		{
			//loop through all pixels of this column
			//(backwards: topmost pixel into least significant bit)
			for( Y = BmDrvInternalPictureSizeY - 1; Y >= 0; Y-- )
			{
				Val <<= 1;
				if( pPicture->Pixels[Y][X] >= PixelValue )
					Val |= 1;
			}
		}

		//output the pixel-value for this column
		outb( Val, BmDrvIoAddr );

		//output a clock-pulse (on pin -nSTROBE)
		outb( 0x1A, BmDrvIoAddr + 2 ); //bidirectional: off, irq: on, -nSTROBE=0, -nAUTOLF=1, nINIT=0, -nSLCTIN=1
		outb( 0x1B, BmDrvIoAddr + 2 ); //bidirectional: off, irq: on, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=1
	} //for( X ...

	//output a load-pulse (on pin -nAUTOLF)
	outb( 0x19, BmDrvIoAddr + 2 ); //bidirectional: off, irq: on, -nSTROBE=1, -nAUTOLF=0, nINIT=0, -nSLCTIN=1
	outb( 0x1B, BmDrvIoAddr + 2 ); //bidirectional: off, irq: on, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=1
}

//input the buttons of the controllers
void BmDrvButtonsIn( struct tBmDrvInternalControllerButtons * pButtonHyst )
{
	//local variables
	int Btn, Ctl;
	unsigned char Val, BtnState[BmDrvInternalControllerCnt];

	//output a load-pulse (on pin -nSLCTIN)
	outb( 0x13, BmDrvIoAddr + 2 ); //bidirectional: off, irq: on, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=0
	outb( 0x1B, BmDrvIoAddr + 2 ); //bidirectional: off, irq: on, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=1

	//read in all buttons
	for( Btn = 0; Btn < BmDrvInternalButtonCnt; Btn++ )
	{
		//read in this button of all controllers
		Val = inb( BmDrvIoAddr + 1 ); //read in 3 times
		Val = inb( BmDrvIoAddr + 1 ); //to give pad time
		Val = inb( BmDrvIoAddr + 1 ); //to output data
		//only interested in: nERROR, SELECT, PPROUT, -BUSY
		Val &= 0xB8;
		//-BUSY has got inverted hardware, but our buttons are active low
		Val ^= 0x38; //so invert: nERROR, SELECT, PPROUT
		//now pressed buttons are 1

		//load buttons into registers (BmDrvInternalControllerCount must be 4)
		BtnState[0] = (Val >> 3) & 0x01; //nERROR is controller 0
		BtnState[1] = (Val >> 4) & 0x01; //SELECT is controller 1
		BtnState[2] = (Val >> 5) & 0x01; //PPROUT is controller 2
		BtnState[3] = (Val >> 7) & 0x01; //-BUSY is controller 3

		//calculate new hysteresis-state
		for( Ctl = 0; Ctl < BmDrvInternalControllerCnt; Ctl++ )
		{
			if( BtnState[Ctl] ) //button is pressed now
			{
				//increase hysteresis if not at maximum
				if( pButtonHyst->Buttons[Ctl][Btn] < BmDrvInternalButtonHyst )
					(pButtonHyst->Buttons[Ctl][Btn])++;
			}
			else //button is released now
			{
				//decrease hysteresis if not at minimum
				if( pButtonHyst->Buttons[Ctl][Btn] > 0 )
					(pButtonHyst->Buttons[Ctl][Btn])--;
			}
		}

		//output a clock-pulse (on pin nINIT)
		outb( 0x1F, BmDrvIoAddr + 2 ); //bidirectional: off, irq: on, -nSTROBE=1, -nAUTOLF=1, nINIT=1, -nSLCTIN=1
		outb( 0x1B, BmDrvIoAddr + 2 ); //bidirectional: off, irq: on, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=1
	} //for( Btn ...

	//generate button states out of hysteresis-counters
	for( Ctl = 0; Ctl < BmDrvInternalControllerCnt; Ctl++ )
	{
		for( Btn = 0; Btn < BmDrvInternalButtonCnt; Btn++ )
		{
			//remember last button state
			BmDrvLastButtonState.Buttons[Ctl][Btn] = BmDrvButtonState.Buttons[Ctl][Btn];
			//get current button state
			if( BmDrvButtonHyst.Buttons[Ctl][Btn] > BmDrvInternalButtonHyst / 2 )
				BmDrvButtonState.Buttons[Ctl][Btn] = 1;
			else
				BmDrvButtonState.Buttons[Ctl][Btn] = 0;
			//button was pressed
			if( BmDrvButtonState.Buttons[Ctl][Btn] && ! BmDrvLastButtonState.Buttons[Ctl][Btn] )
				//write event into read-buffer
				BmDrvReadBufferWrite( 0xFF, 0x01, Ctl, Btn );
			//button was released
			if( ! BmDrvButtonState.Buttons[Ctl][Btn] && BmDrvLastButtonState.Buttons[Ctl][Btn] )
				//write event into read-buffer
				BmDrvReadBufferWrite( 0xFF, 0x00, Ctl, Btn );
		}
	}
	//set change-flag for button state
	BmDrvButtonStateChanged = 1;
}

//the interrupt procedure
void BmDrvInt( int IrqNo, void * DevId, struct pt_regs * PtRegs )
{
	//check IRQ-number and device-id
	if( IrqNo != BmDrvIrqNo || DevId != BmDrvDevId )
		return;

	//start of a new output-cycle
	if( BmDrvOutputStep == 0 )
		//copy next picture (that belongs to the interrupt) to current picture
		memcpy( &BmDrvCurrentPicture, &BmDrvNextPictures[BmDrvNextPictureInterruptIndex], sizeof( struct tBmDrvInternalPicture ) );

	//output the current frame of the current picture
	BmDrvFrameOut( &BmDrvCurrentPicture, BmDrvOutputStepPixelVal[BmDrvOutputStep] );

	//next output step
	BmDrvOutputStep++;
	if( BmDrvOutputStep >= BmDrvOutputSteps )
		BmDrvOutputStep = 0;

	//read in buttons
	BmDrvButtonsIn( &BmDrvButtonHyst );
}

//sleep the specified number of milliseconds (no busy sleep, do not call fron interrupt)
void SleepMilliseconds( int Milliseconds )
{
	//do not do anything if time is zero or less
	if( Milliseconds <= 0 )
		return;

	{
		//define an own wait queue for this purpose
		wait_queue_head_t WaitQueueHead;

		//initialize wait queue
		init_waitqueue_head( &WaitQueueHead );

		//sleep
		sleep_on_timeout( &WaitQueueHead, Milliseconds * HZ / 1000 );
	}
}

//a process is opening the output device
int BmDrvOutFileOpen( struct inode * Inode, struct file * File )
{
	//if output device is already open
	if( BmDrvOutIsOpen )
		//open failed
		return -EBUSY;

	//output device is now open
	BmDrvOutIsOpen = 1;
	BmDrvOutFile = File; //remember pointer to file-struct
	//module is now used once more
	MOD_INC_USE_COUNT;

	//empty next picture that belongs not to the interrupt
	memset( &BmDrvNextPictures[1 - BmDrvNextPictureInterruptIndex], 0, sizeof( struct tBmDrvInternalPicture ) );
	//make the new next picture the one that belongs to the interrupt
	//(because the assignment is an atomic instruction nothing can go wrong)
	//(the interrupt only reads this variable)
	//(this line can't be executed during execution of the interrupt)
	BmDrvNextPictureInterruptIndex = 1 - BmDrvNextPictureInterruptIndex;

	//start with a new frame when writing to the device
	BmDrvWriteSizeX = -1; //no size yet
	BmDrvWriteSizeY = -1;
	BmDrvWritePosX = 0; //start with first pixel
	BmDrvWritePosY = 0;
	memset( &BmDrvWritePicture, 0, sizeof( struct tBmDrvInternalPicture ) ); //empty picture
	BmDrvWriteDelay = -1; //no delay yet

	//open succeeded
	return 0;  
}

//a process is opening the input device
int BmDrvInFileOpen( struct inode * Inode, struct file * File )
{
	//if input device is already open
	if( BmDrvInIsOpen )
		//open failed
		return -EBUSY;

	//input device is now open
	BmDrvInIsOpen = 1;
	BmDrvInFile = File; //remember pointer to file-struct
	//module is now used once more
	MOD_INC_USE_COUNT;

	//open succeeded
	return 0;  
}

//a process is opening the device
int BmDrvFileOpen( struct inode * Inode, struct file * File )
{
	//look at minor device number
	switch( MINOR( Inode->i_rdev ) )
	{
		case BmDrvOutDevNoMinor:
			return BmDrvOutFileOpen( Inode, File );
		case BmDrvInDevNoMinor:
			return BmDrvInFileOpen( Inode, File );
	}

	//no such device
	return -ENODEV;
}

//a process is closing the output device
int BmDrvOutFileClose( struct inode * Inode, struct file * File )
{
	//if output device is not open
	if( !BmDrvOutIsOpen )
		//close failed
		return -EBADF;

	//copy standard picture into next picture that belongs not to the interrupt
	memcpy( &BmDrvNextPictures[1 - BmDrvNextPictureInterruptIndex], &BmDrvStdPicture, sizeof( struct tBmDrvInternalPicture ) );
	//make the new next picture the one that belongs to the interrupt
	//(because the assignment is an atomic instruction nothing can go wrong)
	//(the interrupt only reads this variable)
	//(this line can't be executed during execution of the interrupt)
	BmDrvNextPictureInterruptIndex = 1 - BmDrvNextPictureInterruptIndex;
 
	//output device is now closed
	BmDrvOutIsOpen = 0;
	BmDrvOutFile = NULL;
	//module is now used once less
	MOD_DEC_USE_COUNT;

	//close succeeded
	return 0;  
}

//a process is closing the input device
int BmDrvInFileClose( struct inode * Inode, struct file * File )
{
	//if input device is not open
	if( !BmDrvInIsOpen )
		//close failed
		return -EBADF;

	//input device is now closed
	BmDrvInIsOpen = 0;
	BmDrvInFile = NULL;
	//module is now used once less
	MOD_DEC_USE_COUNT;

	//close succeeded
	return 0;  
}

//a process is closing the device
int BmDrvFileClose( struct inode * Inode, struct file * File )
{
	//look at minor device number
	switch( MINOR( Inode->i_rdev ) )
	{
		case BmDrvOutDevNoMinor:
			return BmDrvOutFileClose( Inode, File );
		case BmDrvInDevNoMinor:
			return BmDrvInFileClose( Inode, File );
	}

	//no such device
	return -ENODEV;
}

//copy version-info to user
int BmDrvIoctlGetVersion( struct tBmDrvVersion * pBmDrvVersion )
{
	//local variables
	struct tBmDrvVersion Ver;
 
	//test pointer
	if( verify_area( VERIFY_WRITE, pBmDrvVersion, sizeof( struct tBmDrvVersion ) ) != 0 )
		//pointer not okay
		return -EFAULT;

	//fill local structure
	strcpy( Ver.Name, BmDrvDevName ); //char[32] is enough
	Ver.Major = BmDrvDevVerMaj;
	Ver.Minor = BmDrvDevVerMin;
	Ver.Revision = BmDrvDevVerRev;

	//copy structure to user-space
	copy_to_user( pBmDrvVersion, &Ver, sizeof( struct tBmDrvVersion ) );

	//success
	return 0;
}

//copy picture-size to user
int BmDrvIoctlGetPictureSize( struct tBmDrvPictureSize * pBmDrvPictureSize )
{
	//local variables
	struct tBmDrvPictureSize Size;
 
	//test pointer
	if( verify_area( VERIFY_WRITE, pBmDrvPictureSize, sizeof( struct tBmDrvPictureSize ) ) != 0 )
		//pointer not okay
		return -EFAULT;

	//fill local structure
	Size.SizeX = BmDrvInternalPictureSizeX;
	Size.SizeY = BmDrvInternalPictureSizeY;

	//copy structure to user-space
	copy_to_user( pBmDrvPictureSize, &Size, sizeof( struct tBmDrvPictureSize ) );

	//success
	return 0;
}

//set the picture
int BmDrvIoctlSetPicture( struct tBmDrvPicture * pBmDrvPicture )
{
	//local variables
	struct tBmDrvPicture Picture;

	//test pointer
	if( verify_area( VERIFY_READ, pBmDrvPicture, sizeof( struct tBmDrvPicture ) ) != 0 )
		//pointer not okay
		return -EFAULT;

	//copy structure from user-space
	copy_from_user( &Picture, pBmDrvPicture, sizeof( struct tBmDrvPicture ) );

	//check picture size
	if( Picture.SizeX != BmDrvInternalPictureSizeX || Picture.SizeY != BmDrvInternalPictureSizeY )
		//picture size is invalid
		return BmDrvErrPictureSize;

	//here we know that:
	//  SizeX == BmDrvInternalPictureSizeX, SizeY == BmDrvInternalPictureSizeY
	//and because of this, we know:
	//  SizeX * SizeY == sizeof( struct tBmDrvInternalPicture )

	//test pixel-buffer
	if( verify_area( VERIFY_READ, Picture.pPixels, sizeof( struct tBmDrvInternalPicture ) ) != 0 )
		//buffer is invalid
		return BmDrvErrPixelBuffer;

	//copy pixels into next picture that belongs not to the interrupt
	//(pixels are in same order as in internal buffer)
	copy_from_user( &BmDrvNextPictures[1 - BmDrvNextPictureInterruptIndex], Picture.pPixels, sizeof( struct tBmDrvInternalPicture ) );
	//make the new next picture the one that belongs to the interrupt
	//(because the assignment is an atomic instruction nothing can go wrong)
	//(the interrupt only reads this variable)
	//(this line can't be executed during execution of the interrupt)
	BmDrvNextPictureInterruptIndex = 1 - BmDrvNextPictureInterruptIndex;

	//success
	return 0;
}

//get number of controllers and buttons
int BmDrvIoctlGetButtonInfo( struct tBmDrvButtonInfo * pBmDrvButtonInfo )
{
	//local variables
	struct tBmDrvButtonInfo Info;
 
	//test pointer
	if( verify_area( VERIFY_WRITE, pBmDrvButtonInfo, sizeof( struct tBmDrvButtonInfo ) ) != 0 )
		//pointer not okay
		return -EFAULT;

	//fill local structure
	Info.ControllerCnt = BmDrvInternalControllerCnt;
	Info.ButtonCnt = BmDrvInternalButtonCnt;

	//copy structure to user-space
	copy_to_user( pBmDrvButtonInfo, &Info, sizeof( struct tBmDrvButtonInfo ) );

	//success
	return 0;
}

//get state of all buttons
int BmDrvIoctlGetButtonState( struct tBmDrvButtonState * pBmDrvButtonState )
{
	//local variables
	struct tBmDrvButtonState State;
	int I;
	struct tBmDrvInternalControllerButtons ButtonState;
 
	//test pointer
	if( verify_area( VERIFY_READ, pBmDrvButtonState, sizeof( struct tBmDrvButtonState ) ) != 0 )
		//pointer not okay
		return -EFAULT;

	//copy structure from user-space
	copy_from_user( &State, pBmDrvButtonState, sizeof( struct tBmDrvButtonState ) );

	//check number of controllers and buttons
	if( State.ControllerCnt != BmDrvInternalControllerCnt || State.ButtonCnt != BmDrvInternalButtonCnt )
		//number of controllers and/or buttons is invalid
		return BmDrvErrCtlBtnCnt;

	//here we know that:
	//  ControllerCnt == BmDrvInternalControllerCnt, ButtonCnt == BmDrvInternalButtonCnt
	//and because of this, we know:
	//  ControllerCnt * ButtonCnt == sizeof( struct tBmDrvInternalControllerButtons )

	//test button-buffer
	if( verify_area( VERIFY_WRITE, State.pButtons, sizeof( struct tBmDrvInternalControllerButtons ) ) != 0 )
		//buffer is invalid
		return BmDrvErrBtnBuffer;

	//get button state
	for( I = 0; I < 8; I++ ) //try to read a consistent state at max. 8 times
	{
		//reset change flag
		BmDrvButtonStateChanged = 0;
		//copy state to local buffer
		memcpy( &ButtonState, &BmDrvButtonState, sizeof( struct tBmDrvInternalControllerButtons ) );
		//change flag not set
		if( ! BmDrvButtonStateChanged )
			//no interrupt while reading, so state is consistent
			break;
	}

	//copy button states into user-buffer
	//(pixels are in same order as in internal buffer)
	copy_to_user( State.pButtons, &ButtonState, sizeof( struct tBmDrvInternalControllerButtons ) );

	//success
	return 0;
}

//a process is doing an ioctl on the output device
int BmDrvOutFileIoctl( struct inode * Inode, struct file * File, unsigned int IoctlNo, unsigned long IoctlParam )
{
	//if output device is not open
	if( !BmDrvOutIsOpen )
		//ioctl failed
		return -EBADF;

	//choose ioctl
	switch( IoctlNo )
	{
		//get version
		case BmDrvIoctlNoGetVersion:
			return BmDrvIoctlGetVersion( (struct tBmDrvVersion *)IoctlParam );
			break;

		//get picture size
		case BmDrvIoctlNoGetPictureSize:
			return BmDrvIoctlGetPictureSize( (struct tBmDrvPictureSize *)IoctlParam );
			break;

		//set picture
		case BmDrvIoctlNoSetPicture:
			return BmDrvIoctlSetPicture( (struct tBmDrvPicture *)IoctlParam );
			break;

		//unknown ioctl
		default:
			return BmDrvErrIoctlNo;
	}
}

//a process is doing an ioctl on the input device
int BmDrvInFileIoctl( struct inode * Inode, struct file * File, unsigned int IoctlNo, unsigned long IoctlParam )
{
	//if input device is not open
	if( !BmDrvInIsOpen )
		//ioctl failed
		return -EBADF;

	//choose ioctl
	switch( IoctlNo )
	{
		//get version
		case BmDrvIoctlNoGetVersion:
			return BmDrvIoctlGetVersion( (struct tBmDrvVersion *)IoctlParam );
			break;

		//get number of controllers and buttons
		case BmDrvIoctlNoGetButtonInfo:
			return BmDrvIoctlGetButtonInfo( (struct tBmDrvButtonInfo *)IoctlParam );
			break;

		//get state of all buttons
		case BmDrvIoctlNoGetButtonState:
			return BmDrvIoctlGetButtonState( (struct tBmDrvButtonState *)IoctlParam );
			break;

		//unknown ioctl
		default:
			return BmDrvErrIoctlNo;
	}
}

//a process is doing an ioctl on the device
int BmDrvFileIoctl( struct inode * Inode, struct file * File, unsigned int IoctlNo, unsigned long IoctlParam )
{
	//look at minor device number
	switch( MINOR( Inode->i_rdev ) )
	{
		case BmDrvOutDevNoMinor:
			return BmDrvOutFileIoctl( Inode, File, IoctlNo, IoctlParam );
		case BmDrvInDevNoMinor:
			return BmDrvInFileIoctl( Inode, File, IoctlNo, IoctlParam );
	}

	//no such device
	return -ENODEV;
}

//a process is writing to the output device
ssize_t BmDrvOutFileWrite( struct file * File, const char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//local variables
	int I;

	//if output device is not open
	if( !BmDrvOutIsOpen )
		//write failed
		return -EBADF;

	//test buffer
	if( verify_area( VERIFY_READ, BufferPtr, BufferLen ) != 0 )
		//buffer not okay
		return -EFAULT;

	//process bytes in buffer
	for( I = 0; I < BufferLen; I++ )
	{
		//size-x
		if( BmDrvWriteSizeX <= -1 )
		{
			//read size-x
			BmDrvWriteSizeX = (unsigned char)BufferPtr[I];
		}
		//size-y
		else if( BmDrvWriteSizeY <= -1 )
		{
			//read size-y
			BmDrvWriteSizeY = (unsigned char)BufferPtr[I];
			//calculate number of pixels to ignore (on left and on top) and offset of picture
			if( BmDrvWriteSizeX > BmDrvInternalPictureSizeX )
			{
				BmDrvWriteVisibleL = (BmDrvWriteSizeX - BmDrvInternalPictureSizeX) / 2;
				BmDrvWriteVisibleR = BmDrvWriteVisibleL + BmDrvInternalPictureSizeX - 1;
				BmDrvWriteOffsetX = 0;
			}
			else
			{
				BmDrvWriteVisibleL = 0;
				BmDrvWriteVisibleR = BmDrvWriteSizeX - 1;
				BmDrvWriteOffsetX = (BmDrvInternalPictureSizeX - BmDrvWriteSizeX) / 2;
			}
			if( BmDrvWriteSizeY > BmDrvInternalPictureSizeY )
			{
				BmDrvWriteVisibleT = (BmDrvWriteSizeY - BmDrvInternalPictureSizeY) / 2;
				BmDrvWriteVisibleB = BmDrvWriteVisibleT + BmDrvInternalPictureSizeY - 1;
				BmDrvWriteOffsetY = 0;
			}
			else
			{
				BmDrvWriteVisibleT = 0;
				BmDrvWriteVisibleB = BmDrvWriteSizeY - 1;
				BmDrvWriteOffsetY = (BmDrvInternalPictureSizeY - BmDrvWriteSizeY) / 2;
			}
		}
		//pixels
		else if( BmDrvWritePosY < BmDrvWriteSizeY )
		{
			//pixel within visible area of picture
			if( BmDrvWritePosX >= BmDrvWriteVisibleL && BmDrvWritePosX <= BmDrvWriteVisibleR
			 && BmDrvWritePosY >= BmDrvWriteVisibleT && BmDrvWritePosY <= BmDrvWriteVisibleB )
			{
				//write pixel into buffer
				BmDrvWritePicture.Pixels
						[BmDrvWritePosY - BmDrvWriteVisibleT + BmDrvWriteOffsetY]
						[BmDrvWritePosX - BmDrvWriteVisibleL + BmDrvWriteOffsetX]
						= (unsigned char)BufferPtr[I];
			}
			//next pixel
			BmDrvWritePosX++;
			if( BmDrvWritePosX >= BmDrvWriteSizeX )
			{
				BmDrvWritePosX = 0;
				BmDrvWritePosY++;
			}
		}
		//delay
		else
		{
			//read delay
			BmDrvWriteDelay = (unsigned char)BufferPtr[I];
			//now frame is complete
			//write frame into buffer for next picture
			memcpy( &BmDrvNextPictures[1 - BmDrvNextPictureInterruptIndex], &BmDrvWritePicture, sizeof( struct tBmDrvInternalPicture ) );
			//make the new next picture the one that belongs to the interrupt
			//(because the assignment is an atomic instruction nothing can go wrong)
			//(the interrupt only reads this variable)
			//(this line can't be executed during execution of the interrupt)
			BmDrvNextPictureInterruptIndex = 1 - BmDrvNextPictureInterruptIndex;
			//wait for delay to pass
			SleepMilliseconds( BmDrvWriteDelay * 5 );
			//start new frame
			BmDrvWriteSizeX = -1; //no size yet
			BmDrvWriteSizeY = -1;
			BmDrvWritePosX = 0; //start with first pixel
			BmDrvWritePosY = 0;
			memset( &BmDrvWritePicture, 0, sizeof( struct tBmDrvInternalPicture ) ); //empty picture
			BmDrvWriteDelay = -1; //no delay yet
		}
	} //for( I ...

	//all bytes read
	return BufferLen;
}

//a process is writing to the input device
ssize_t BmDrvInFileWrite( struct file * File, const char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//if input device is not open
	if( !BmDrvInIsOpen )
		//write failed
		return -EBADF;

	//test buffer
	if( verify_area( VERIFY_READ, BufferPtr, BufferLen ) != 0 )
		//buffer not okay
		return -EFAULT;

	//ignore all bytes
	return BufferLen;
}

//a process is writing to the device
ssize_t BmDrvFileWrite( struct file * File, const char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//writing to output device
	if( File == BmDrvOutFile )
		return BmDrvOutFileWrite( File, BufferPtr, BufferLen, OffsetPtr );

	//writing to input device
	if( File == BmDrvInFile )
		return BmDrvInFileWrite( File, BufferPtr, BufferLen, OffsetPtr );

	//write failed
	return -EBADF;
}

//a process is reading from the output device
//reading from this device is always nonblocking - so when read returning 0 does not mean EOF
ssize_t BmDrvOutFileRead( struct file * File, char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//if output device is not open
	if( !BmDrvOutIsOpen )
		//read failed
		return -EBADF;

	//test buffer
	if( verify_area( VERIFY_WRITE, BufferPtr, BufferLen ) != 0 )
		//buffer not okay
		return -EFAULT;

	//nothing written
	return 0;
}

//a process is reading from the input device
//(for format, see declaraiton of BmDrvReadBuffer)
//reading from this device is always nonblocking - so when read returning 0 does not mean EOF
ssize_t BmDrvInFileRead( struct file * File, char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//local variables
	int Count;

	//if input device is not open
	if( !BmDrvInIsOpen )
		//read failed
		return -EBADF;

	//test buffer
	if( verify_area( VERIFY_WRITE, BufferPtr, BufferLen ) != 0 )
		//buffer not okay
		return -EFAULT;

	//nothing read from this device yet
	Count = 0;

	//put left bytes from last read into buffer, as long as space in buffer
	while( BmDrvReadChrBufferCnt > 0 && BufferLen - Count > 0 )
	{
		//put first byte into buffer
		BufferPtr[Count] = BmDrvReadChrBuffer[0];
		//next byte in buffer
		Count++;
		//remove a byte from the character buffer with left characters
		BmDrvReadChrBuffer[0] = BmDrvReadChrBuffer[1];
		BmDrvReadChrBuffer[1] = BmDrvReadChrBuffer[2];
		BmDrvReadChrBuffer[2] = BmDrvReadChrBuffer[3];
		BmDrvReadChrBufferCnt--;
	}

	//put entire actions into buffer, as long as space in buffer
	while( BufferLen - Count >= 4 )
	{
		//read from read buffer (4 bytes)
		if( BmDrvReadBufferRead( (unsigned char *)BufferPtr + Count ) != 0 )
			//read buffer empty
			break;
		//count read bytes
		Count += 4;
	}

	//space left in buffer
	if( BufferLen - Count > 0 )
	{
		//character buffer empty
		if( BmDrvReadChrBufferCnt == 0 )
			//get an entire action from the buffer and put it into the character buffer (4 bytes)
			if( BmDrvReadBufferRead( (unsigned char *)BmDrvReadChrBuffer ) == 0 )
				BmDrvReadChrBufferCnt = 4;
		//put bytes from character buffer into buffer, as long as space in buffer
		while( BmDrvReadChrBufferCnt > 0 && BufferLen - Count > 0 )
		{
			//put first byte into buffer
			BufferPtr[Count] = BmDrvReadChrBuffer[0];
			//next byte in buffer
			Count++;
			//remove a byte from the character buffer with left characters
			BmDrvReadChrBuffer[0] = BmDrvReadChrBuffer[1];
			BmDrvReadChrBuffer[1] = BmDrvReadChrBuffer[2];
			BmDrvReadChrBuffer[2] = BmDrvReadChrBuffer[3];
			BmDrvReadChrBufferCnt--;
		}
	}

	//return number of bytes written
	return Count;
}

//a process is reading from the device
//reading from this device is always nonblocking - so when read returning 0 does not mean EOF
ssize_t BmDrvFileRead( struct file * File, char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//reading from output device
	if( File == BmDrvOutFile )
		return BmDrvOutFileRead( File, BufferPtr, BufferLen, OffsetPtr );

	//reading from input device
	if( File == BmDrvInFile )
		return BmDrvInFileRead( File, BufferPtr, BufferLen, OffsetPtr );

	//read failed
	return -EBADF;
}

//structure for supported operations
struct file_operations BmDrvFileOperations =
{
#if LINUX_VERSION_CODE >= 0x20400 //kernel-version 2.4
	NULL, //owner
#endif
	NULL, //llseek
	BmDrvFileRead, //read
	BmDrvFileWrite, //write
	NULL, //readdir
	NULL, //poll
	BmDrvFileIoctl, //io-control
	NULL, //mmap
	BmDrvFileOpen, //open
	NULL, //flush
	BmDrvFileClose, //release
};

//module is being loaded
int init_module( )
{
	//local variables
	int I;
	struct resource * ResPtr;

	//short message
	printk( "bmdrv: BLINKENmini driver for the parallel port (version %d.%d.%d)\n", BmDrvDevVerMaj, BmDrvDevVerMin, BmDrvDevVerRev );
	printk( "bmdrv: done in 2002 by sphaera & 1stein\n" );
	printk( "bmdrv: http://blinkenmini.1stein.no-ip.com/\n" );
	printk( "bmdrv: distributed under GNU Public License - no warranty\n" );
	printk( "bmdrv: trying to load (io=0x%X irq=%d)...\n", io, irq );

	//check startup parameters - io-address
	for( I = 0; I < count( BmDrvIoAddrs ); I++ )
		if( BmDrvIoAddrs[I] == (unsigned short)io )
			break;
	//invalid io-address
	if( I >= count( BmDrvIoAddrs ) )
	{
		//error message
		printk( "bmdrv: io=0x%X is not valid\n", io );
		printk( "bmdrv: valid settings for io are:" );
		for( I = 0; I < count( BmDrvIoAddrs ); I++ )
			printk( " 0x%X", BmDrvIoAddrs[I] );
		printk( "\n" );
		//init failed
		return -1;
	}
	//remember used io-address
	BmDrvIoAddr = BmDrvIoAddrs[I];

	//check startup parameters - irq-number
	for( I = 0; I < count( BmDrvIrqNos ); I++ )
		if( BmDrvIrqNos[I] == (unsigned char)irq )
			break;
	//invalid irq-number
	if( I >= count( BmDrvIrqNos ) )
	{
		//error message
		printk( "bmdrv: irq=0x%X is not valid\n", irq );
		printk( "bmdrv: valid settings for irq are:" );
		for( I = 0; I < count( BmDrvIrqNos ); I++ )
			printk( " %d", BmDrvIrqNos[I] );
		printk( "\n" );
		//init failed
		return -1;
	}
	//remember used irq-number
	BmDrvIrqNo = BmDrvIrqNos[I];

	//output and input device is not opend by a process yet
	BmDrvOutIsOpen = 0;
	BmDrvOutFile = NULL;
	BmDrvInIsOpen = 0;
	BmDrvInFile = NULL;

	//set the next picture to the standard picture
	//(nothing can go wrong, because interrupt is not active)
	memcpy( &BmDrvNextPictures[0], &BmDrvStdPicture, sizeof( struct tBmDrvInternalPicture ) ); //set picture 0
	BmDrvNextPictureInterruptIndex = 0; //picture 0 belongs to interrupt
	//start outputting the next picture
	//(nothing can go wrong, because interrupt is not active)
	BmDrvOutputStep = 0;

	//all buttons of all controllers are not pressed
	//(nothing can go wrong, because interrupt is not active)
	memset( &BmDrvButtonHyst, 0, sizeof( struct tBmDrvInternalControllerButtons ) );
	//last and current button state: all buttons released
	//(nothing can go wrong, because interrupt is not active)
	memset( &BmDrvLastButtonState, 0, sizeof( struct tBmDrvInternalControllerButtons ) );
	memset( &BmDrvButtonState, 0, sizeof( struct tBmDrvInternalControllerButtons ) );

	//read-buffer is empty
	BmDrvReadBufferStart = 0;
	BmDrvReadBufferEnd = 0;
	//no left characters from last read
	BmDrvReadChrBufferCnt = 0;

	//register device
	I = register_chrdev( BmDrvDevNoMajor, BmDrvDevName, &BmDrvFileOperations );
	//an error occured
	if( I < 0 )
	{
		//error-message
		printk( "bmdrv: register_chrdev failed with %d\n", I );
		//init failed
		return -1;
	}

	//register io-ports
	ResPtr = request_region( BmDrvIoAddr, 3, BmDrvDevName );
	//not free
	if( ResPtr == NULL )
	{
		//error-message
		printk( "bmdrv: request_region failed: io-ports not free\n" );
		//unregister device
		unregister_chrdev( BmDrvDevNoMajor, BmDrvDevName );
		//init failed
		return -1;
	}

	//setup hardware
	outb( 0x00, BmDrvIoAddr ); //set data-bits to 0
	outb( 0x1B, BmDrvIoAddr + 2 ); //bidirectional: off, irq: on, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=1

	//register hardware-interrupt (not shareable)
	I = request_irq( BmDrvIrqNo, BmDrvInt, 0, BmDrvDevName, BmDrvDevId );
	//an error occured
	if( I != 0 )
	{
		//error-message
		printk( "bmdrv: request_irq failed with %d\n", I );
		//unregister io-ports
		release_region( BmDrvIoAddr, 3 );
		//unregister device
		unregister_chrdev( BmDrvDevNoMajor, BmDrvDevName );
		//init failed
		return -1;
	}

	//init succeeded
	printk( "bmdrv: loaded (io=0x%X irq=%d)...\n", BmDrvIoAddr, BmDrvIrqNo );
	return 0;
}

//module is being unloaded
void cleanup_module( )
{
	//empty next picture that belongs not to the interrupt
	memset( &BmDrvNextPictures[1 - BmDrvNextPictureInterruptIndex], 0, sizeof( struct tBmDrvInternalPicture ) );
	//make the new next picture the one that belongs to the interrupt
	//(because the assignment is an atomic instruction nothing can go wrong)
	//(the interrupt only reads this variable)
	//(this line can't be executed during execution of the interrupt)
	BmDrvNextPictureInterruptIndex = 1 - BmDrvNextPictureInterruptIndex;

	//unregister hardware-interrupt
	free_irq( BmDrvIrqNo, BmDrvDevId );

	//output empty frame
	BmDrvFrameOut( NULL, 0 );

	//reset hardware
	outb( 0x00, BmDrvIoAddr ); //set data-bits to 0
	outb( 0x0B, BmDrvIoAddr + 2 ); //bidirectional: off, irq: off, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=1

	//unregister io-ports
	release_region( BmDrvIoAddr, 3 );

  	//unregister device
	unregister_chrdev( BmDrvDevNoMajor, BmDrvDevName );

	//done
	printk( "bmdrv: unloaded...\n" );
}

