/* kernel driver for BlinkenLeds (compatible to BLINKENmini)
 *  - character device kernel module for output to BlinkenLeds using the parallel port
 * version 1.1.92 date 2006-01-08
 * Copyright (C) 2002-2006 sphaera & 1stein (http://blinkenmini.schuermans.info/)
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/ioport.h>
#include <linux/fs.h>
#if LINUX_VERSION_CODE >= 0x20600 //kernel-version 2.6
#else
#include <linux/wrapper.h>
#endif
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/param.h>

//number of elements in an array
#define count( array ) (sizeof( (array) ) / sizeof( (array)[0] ))

//version information for insmod
char version[] = UTS_RELEASE;

//startup parameters
int io = 0;
MODULE_PARM( io, "i" );

//include bmdrv-header-file
#include "bmdrv_leds.h"

//global constants concerning the module
#define BmDrvDevName "bmdrv_leds" //the name of this device (max. 31 chars allowed)
#define BmDrvDevId ((void *)(('b'<<24)+('m'<<16)+('d'<<8)+'r')) //the id of this device
//#error new_ver: UPDATE numerical version information!!!
#define BmDrvDevVerMaj 1 //version of this device
#define BmDrvDevVerMin 1
#define BmDrvDevVerRev 92

//global types
#define BmDrvInternalPictureSizeX 18 //dimensions of an internal BM-picture
#define BmDrvInternalPictureSizeY 8
typedef struct tBmDrvInternalPicture //an internal BM-picture
{
	unsigned char Pixels[BmDrvInternalPictureSizeY][BmDrvInternalPictureSizeX];
} stBmDrvInternalPicture;
#define BmDrvInternalButtonCnt 16 //number of buttons per controller
#define BmDrvInternalControllerCnt 4 //number of controllers
#define BmDrvInternalButtonHyst 16 //width of hysteresis
typedef struct tBmDrvInternalControllerButtons //state of the internal controller buttons
{
	unsigned char Buttons[BmDrvInternalControllerCnt][BmDrvInternalButtonCnt];
} stBmDrvInternalControllerButtons;

//global variables
int BmDrvOutIsOpen = 0; //true if the output device is opened by a process
struct file * BmDrvOutFile; //file-struct of output-file (NULL if closed)
unsigned short BmDrvIoAddrs[] = { 0x3BC, 0x378, 0x278 }; //valid io-addresses
unsigned short BmDrvIoAddr; //io-address

//the picture to display
struct tBmDrvInternalPicture BmDrvPicture;

/*
//the standard-picture: diagonal lines
struct tBmDrvInternalPicture BmDrvStdPicture =
{
Pixels:
	{
		{ 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, },
		{ 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, },
		{ 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, },
		{ 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, },
		{ 0x40, 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, },
		{ 0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, },
		{ 0x60, 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, },
		{ 0x70, 0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0, 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, },
 	}
};
*/

/*
//the standard-picture: BM
struct tBmDrvInternalPicture BmDrvStdPicture =
{
Pixels:
	{
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
 	}
};
*/

//the standard-picture: BM with shadow
struct tBmDrvInternalPicture BmDrvStdPicture =
{
Pixels:
	{
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, },
		{ 0x00, 0xFF, 0xFF, 0x40, 0x40, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x40, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0x40, 0x40, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0x40, 0x40, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x40, 0x40, 0x00, 0xFF, 0xFF, 0x40, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x40, },
		{ 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x00, 0x00, 0x00, 0x00, 0x40, 0x40, },
 	}
};

//global variables for writing to the device in bmbm-format
int BmDrvWriteSizeX, BmDrvWriteSizeY; //the size of the frame being being written (-1: size not written yet)
int BmDrvWriteVisibleL, BmDrvWriteVisibleR, BmDrvWriteVisibleT, BmDrvWriteVisibleB; //left, right, top, bottom of visible area of picture
int BmDrvWriteOffsetX, BmDrvWriteOffsetY; //offset of topleft-pixel of written picture from topleft-pixel of display
int BmDrvWritePosX, BmDrvWritePosY; //position of the next pixel that has to be written
struct tBmDrvInternalPicture BmDrvWritePicture; //the picture with the data already written
int BmDrvWriteDelay; //the delay of the frame currently being written (-1: delay not written yet)

//output a frame
//turns on all pixels of BmDrvPicture that have a value >= 0x80
void BmDrvFrameOut( )
{
	//local variables
	int X, Y;
	unsigned char Val, Strobe;

	//loop through all 9 bit groups
	for( X = 1; X >= 0; X-- )
	for( Y = 7; Y >= 0; Y-- )
	{
		//calculate value and strobe for this 9 pixels
		Val = 0x00;
		if( BmDrvPicture.Pixels[Y][ 0 + X] >= 0x80 )
			Val |= 0x01;
		if( BmDrvPicture.Pixels[Y][ 2 + X] >= 0x80 )
			Val |= 0x02;
		if( BmDrvPicture.Pixels[Y][ 4 + X] >= 0x80 )
			Val |= 0x04;
		if( BmDrvPicture.Pixels[Y][ 6 + X] >= 0x80 )
			Val |= 0x08;
		if( BmDrvPicture.Pixels[Y][ 8 + X] >= 0x80 )
			Val |= 0x10;
		if( BmDrvPicture.Pixels[Y][10 + X] >= 0x80 )
			Val |= 0x20;
		if( BmDrvPicture.Pixels[Y][12 + X] >= 0x80 )
			Val |= 0x40;
		if( BmDrvPicture.Pixels[Y][14 + X] >= 0x80 )
			Val |= 0x80;
		Strobe = 0x00;
		if( BmDrvPicture.Pixels[Y][16 + X] >= 0x80 )
			Strobe |= 0x01;

		//output the next 9 bits of data
		outb( Val, BmDrvIoAddr );
		outb( 0x03 & ~Strobe, BmDrvIoAddr + 2 ); //bidirectional: off, irq: off, -nSTROBE=!Strobe, -nAUTOLF=1, nINIT=0, -nSLCTIN=0

		//output a clock-pulse (on pin -nAUTOLF)
		outb( 0x01 & ~Strobe, BmDrvIoAddr + 2 ); //bidirectional: off, irq: off, -nSTROBE=!Strobe, -nAUTOLF=0, nINIT=0, -nSLCTIN=0
		outb( 0x03 & ~Strobe, BmDrvIoAddr + 2 ); //bidirectional: off, irq: off, -nSTROBE=!Strobe, -nAUTOLF=1, nINIT=0, -nSLCTIN=0
	} //for( Y ... for( X ...

	//output a load-pulse (on pin nINIT)
	outb( 0x07, BmDrvIoAddr + 2 ); //bidirectional: off, irq: off, -nSTROBE=1, -nAUTOLF=1, nINIT=1, -nSLCTIN=0
	outb( 0x03, BmDrvIoAddr + 2 ); //bidirectional: off, irq: off, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=0
}

//sleep the specified number of milliseconds (no busy sleep, do not call fron interrupt)
void SleepMilliseconds( int Milliseconds )
{
	//do not do anything if time is zero or less
	if( Milliseconds <= 0 )
		return;

	{
		//define an own wait queue for this purpose
		wait_queue_head_t WaitQueueHead;

		//initialize wait queue
		init_waitqueue_head( &WaitQueueHead );

		//sleep
		sleep_on_timeout( &WaitQueueHead, Milliseconds * HZ / 1000 );
	}
}

//a process is opening the output device
int BmDrvOutFileOpen( struct inode * Inode, struct file * File )
{
	//if output device is already open
	if( BmDrvOutIsOpen )
		//open failed
		return -EBUSY;

	//output device is now open
	BmDrvOutIsOpen = 1;
	BmDrvOutFile = File; //remember pointer to file-struct
	//module is now used once more
#if LINUX_VERSION_CODE < 0x20400 //before kernel-version 2.4
	MOD_INC_USE_COUNT;
#endif

	//empty picture
	memset( &BmDrvPicture, 0, sizeof( struct tBmDrvInternalPicture ) );
	//output frame
	BmDrvFrameOut( );

	//start with a new frame when writing to the device
	BmDrvWriteSizeX = -1; //no size yet
	BmDrvWriteSizeY = -1;
	BmDrvWritePosX = 0; //start with first pixel
	BmDrvWritePosY = 0;
	memset( &BmDrvWritePicture, 0, sizeof( struct tBmDrvInternalPicture ) ); //empty picture
	BmDrvWriteDelay = -1; //no delay yet

	//open succeeded
	return 0;  
}

//a process is opening the device
int BmDrvFileOpen( struct inode * Inode, struct file * File )
{
	//look at minor device number
	switch( MINOR( Inode->i_rdev ) )
	{
		case BmDrvOutDevNoMinor:
			return BmDrvOutFileOpen( Inode, File );
	}

	//no such device
	return -ENODEV;
}

//a process is closing the output device
int BmDrvOutFileClose( struct inode * Inode, struct file * File )
{
	//if output device is not open
	if( !BmDrvOutIsOpen )
		//close failed
		return -EBADF;

	//set the picture to the standard picture
	memcpy( &BmDrvPicture, &BmDrvStdPicture, sizeof( struct tBmDrvInternalPicture ) );
	//output frame
	BmDrvFrameOut( );
 
	//output device is now closed
	BmDrvOutIsOpen = 0;
	BmDrvOutFile = NULL;
	//module is now used once less
#if LINUX_VERSION_CODE < 0x20400 //before kernel-version 2.4
	MOD_DEC_USE_COUNT;
#endif

	//close succeeded
	return 0;  
}

//a process is closing the device
int BmDrvFileClose( struct inode * Inode, struct file * File )
{
	//look at minor device number
	switch( MINOR( Inode->i_rdev ) )
	{
		case BmDrvOutDevNoMinor:
			return BmDrvOutFileClose( Inode, File );
	}

	//no such device
	return -ENODEV;
}

//copy version-info to user
int BmDrvIoctlGetVersion( struct tBmDrvVersion * pBmDrvVersion )
{
	//local variables
	struct tBmDrvVersion Ver;
 
	//test pointer
	if( !access_ok( VERIFY_WRITE, pBmDrvVersion, sizeof( struct tBmDrvVersion ) ) )
		//pointer not okay
		return -EFAULT;

	//fill local structure
	strcpy( Ver.Name, BmDrvDevName ); //char[32] is enough
	Ver.Major = BmDrvDevVerMaj;
	Ver.Minor = BmDrvDevVerMin;
	Ver.Revision = BmDrvDevVerRev;

	//copy structure to user-space
	copy_to_user( pBmDrvVersion, &Ver, sizeof( struct tBmDrvVersion ) );

	//success
	return 0;
}

//copy picture-size to user
int BmDrvIoctlGetPictureSize( struct tBmDrvPictureSize * pBmDrvPictureSize )
{
	//local variables
	struct tBmDrvPictureSize Size;
 
	//test pointer
	if( !access_ok( VERIFY_WRITE, pBmDrvPictureSize, sizeof( struct tBmDrvPictureSize ) ) )
		//pointer not okay
		return -EFAULT;

	//fill local structure
	Size.SizeX = BmDrvInternalPictureSizeX;
	Size.SizeY = BmDrvInternalPictureSizeY;

	//copy structure to user-space
	copy_to_user( pBmDrvPictureSize, &Size, sizeof( struct tBmDrvPictureSize ) );

	//success
	return 0;
}

//set the picture
int BmDrvIoctlSetPicture( struct tBmDrvPicture * pBmDrvPicture )
{
	//local variables
	struct tBmDrvPicture Picture;

	//test pointer
	if( !access_ok( VERIFY_READ, pBmDrvPicture, sizeof( struct tBmDrvPicture ) ) )
		//pointer not okay
		return -EFAULT;

	//copy structure from user-space
	copy_from_user( &Picture, pBmDrvPicture, sizeof( struct tBmDrvPicture ) );

	//check picture size
	if( Picture.SizeX != BmDrvInternalPictureSizeX || Picture.SizeY != BmDrvInternalPictureSizeY )
		//picture size is invalid
		return BmDrvErrPictureSize;

	//here we know that:
	//  SizeX == BmDrvInternalPictureSizeX, SizeY == BmDrvInternalPictureSizeY
	//and because of this, we know:
	//  SizeX * SizeY == sizeof( struct tBmDrvInternalPicture )

	//test pixel-buffer
	if( !access_ok( VERIFY_READ, Picture.pPixels, sizeof( struct tBmDrvInternalPicture ) ) )
		//buffer is invalid
		return BmDrvErrPixelBuffer;

	//save frame
	//(pixels are in same order as in internal buffer)
	copy_from_user( &BmDrvPicture, Picture.pPixels, sizeof( struct tBmDrvInternalPicture ) );
	//output frame
	BmDrvFrameOut( );			

	//success
	return 0;
}

//a process is doing an ioctl on the output device
int BmDrvOutFileIoctl( struct inode * Inode, struct file * File, unsigned int IoctlNo, unsigned long IoctlParam )
{
	//if output device is not open
	if( !BmDrvOutIsOpen )
		//ioctl failed
		return -EBADF;

	//choose ioctl
	switch( IoctlNo )
	{
		//get version
		case BmDrvIoctlNoGetVersion:
			return BmDrvIoctlGetVersion( (struct tBmDrvVersion *)IoctlParam );
			break;

		//get picture size
		case BmDrvIoctlNoGetPictureSize:
			return BmDrvIoctlGetPictureSize( (struct tBmDrvPictureSize *)IoctlParam );
			break;

		//set picture
		case BmDrvIoctlNoSetPicture:
			return BmDrvIoctlSetPicture( (struct tBmDrvPicture *)IoctlParam );
			break;

		//unknown ioctl
		default:
			return BmDrvErrIoctlNo;
	}
}

//a process is doing an ioctl on the device
int BmDrvFileIoctl( struct inode * Inode, struct file * File, unsigned int IoctlNo, unsigned long IoctlParam )
{
	//look at minor device number
	switch( MINOR( Inode->i_rdev ) )
	{
		case BmDrvOutDevNoMinor:
			return BmDrvOutFileIoctl( Inode, File, IoctlNo, IoctlParam );
	}

	//no such device
	return -ENODEV;
}

//a process is writing to the output device
ssize_t BmDrvOutFileWrite( struct file * File, const char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//local variables
	int I;

	//if output device is not open
	if( !BmDrvOutIsOpen )
		//write failed
		return -EBADF;

	//test buffer
	if( !access_ok( VERIFY_READ, BufferPtr, BufferLen ) )
		//buffer not okay
		return -EFAULT;

	//process bytes in buffer
	for( I = 0; I < BufferLen; I++ )
	{
		//size-x
		if( BmDrvWriteSizeX <= -1 )
		{
			//read size-x
			BmDrvWriteSizeX = (unsigned char)BufferPtr[I];
		}
		//size-y
		else if( BmDrvWriteSizeY <= -1 )
		{
			//read size-y
			BmDrvWriteSizeY = (unsigned char)BufferPtr[I];
			//calculate number of pixels to ignore (on left and on top) and offset of picture
			if( BmDrvWriteSizeX > BmDrvInternalPictureSizeX )
			{
				BmDrvWriteVisibleL = (BmDrvWriteSizeX - BmDrvInternalPictureSizeX) / 2;
				BmDrvWriteVisibleR = BmDrvWriteVisibleL + BmDrvInternalPictureSizeX - 1;
				BmDrvWriteOffsetX = 0;
			}
			else
			{
				BmDrvWriteVisibleL = 0;
				BmDrvWriteVisibleR = BmDrvWriteSizeX - 1;
				BmDrvWriteOffsetX = (BmDrvInternalPictureSizeX - BmDrvWriteSizeX) / 2;
			}
			if( BmDrvWriteSizeY > BmDrvInternalPictureSizeY )
			{
				BmDrvWriteVisibleT = (BmDrvWriteSizeY - BmDrvInternalPictureSizeY) / 2;
				BmDrvWriteVisibleB = BmDrvWriteVisibleT + BmDrvInternalPictureSizeY - 1;
				BmDrvWriteOffsetY = 0;
			}
			else
			{
				BmDrvWriteVisibleT = 0;
				BmDrvWriteVisibleB = BmDrvWriteSizeY - 1;
				BmDrvWriteOffsetY = (BmDrvInternalPictureSizeY - BmDrvWriteSizeY) / 2;
			}
		}
		//pixels
		else if( BmDrvWritePosY < BmDrvWriteSizeY )
		{
			//pixel within visible area of picture
			if( BmDrvWritePosX >= BmDrvWriteVisibleL && BmDrvWritePosX <= BmDrvWriteVisibleR
			 && BmDrvWritePosY >= BmDrvWriteVisibleT && BmDrvWritePosY <= BmDrvWriteVisibleB )
			{
				//write pixel into buffer
				BmDrvWritePicture.Pixels
						[BmDrvWritePosY - BmDrvWriteVisibleT + BmDrvWriteOffsetY]
						[BmDrvWritePosX - BmDrvWriteVisibleL + BmDrvWriteOffsetX]
						= (unsigned char)BufferPtr[I];
			}
			//next pixel
			BmDrvWritePosX++;
			if( BmDrvWritePosX >= BmDrvWriteSizeX )
			{
				BmDrvWritePosX = 0;
				BmDrvWritePosY++;
			}
		}
		//delay
		else
		{
			//read delay
			BmDrvWriteDelay = (unsigned char)BufferPtr[I];
			//now frame is complete
			//save frame
			memcpy( &BmDrvPicture, &BmDrvWritePicture, sizeof( struct tBmDrvInternalPicture ) );
			//output frame
			BmDrvFrameOut( );			
			//wait for delay to pass
			SleepMilliseconds( BmDrvWriteDelay * 5 );
			//start new frame
			BmDrvWriteSizeX = -1; //no size yet
			BmDrvWriteSizeY = -1;
			BmDrvWritePosX = 0; //start with first pixel
			BmDrvWritePosY = 0;
			memset( &BmDrvWritePicture, 0, sizeof( struct tBmDrvInternalPicture ) ); //empty picture
			BmDrvWriteDelay = -1; //no delay yet
		}
	} //for( I ...

	//all bytes read
	return BufferLen;
}

//a process is writing to the device
ssize_t BmDrvFileWrite( struct file * File, const char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//writing to output device
	if( File == BmDrvOutFile )
		return BmDrvOutFileWrite( File, BufferPtr, BufferLen, OffsetPtr );

	//write failed
	return -EBADF;
}

//a process is reading from the output device
//reading from this device is always nonblocking - so when read returning 0 does not mean EOF
ssize_t BmDrvOutFileRead( struct file * File, char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//if output device is not open
	if( !BmDrvOutIsOpen )
		//read failed
		return -EBADF;

	//test buffer
	if( !access_ok( VERIFY_WRITE, BufferPtr, BufferLen ) )
		//buffer not okay
		return -EFAULT;

	//nothing written
	return 0;
}

//a process is reading from the device
//reading from this device is always nonblocking - so when read returning 0 does not mean EOF
ssize_t BmDrvFileRead( struct file * File, char * BufferPtr, size_t BufferLen, loff_t * OffsetPtr )
{
	//reading from output device
	if( File == BmDrvOutFile )
		return BmDrvOutFileRead( File, BufferPtr, BufferLen, OffsetPtr );

	//read failed
	return -EBADF;
}

//structure for supported operations
struct file_operations BmDrvFileOperations =
{
#if LINUX_VERSION_CODE >= 0x20400 //kernel-version 2.4
  owner: THIS_MODULE,
#endif
  llseek: NULL,
  read: BmDrvFileRead,
#if LINUX_VERSION_CODE >= 0x20600 //kernel-version 2.6
  aio_read: NULL,
#endif
  write: BmDrvFileWrite,
#if LINUX_VERSION_CODE >= 0x20600 //kernel-version 2.6
  aio_write: NULL,
#endif
  readdir: NULL,
  poll: NULL,
  ioctl: BmDrvFileIoctl,
  mmap: NULL,
  open: BmDrvFileOpen,
  flush: NULL,
  release: BmDrvFileClose,
#if LINUX_VERSION_CODE >= 0x20400 //kernel-version 2.4
  fsync: NULL,
#endif
#if LINUX_VERSION_CODE >= 0x20600 //kernel-version 2.6
  aio_fsync: NULL,
#endif
#if LINUX_VERSION_CODE >= 0x20400 //kernel-version 2.4
  fasync: NULL,
  lock: NULL,
  readv: NULL,
  writev: NULL,
#endif
#if LINUX_VERSION_CODE >= 0x20600 //kernel-version 2.6
  sendfile: NULL,
#endif
#if LINUX_VERSION_CODE >= 0x20400 //kernel-version 2.4
  sendpage: NULL,
  get_unmapped_area: NULL,
#endif
};

//module is being loaded
int init_module( )
{
	//local variables
	int I;
	struct resource * ResPtr;

	//short message
	printk( "bmdrv: BLINKENmini compatible BlinkenLeds driver for the parallel port (version %d.%d.%d)\n", BmDrvDevVerMaj, BmDrvDevVerMin, BmDrvDevVerRev );
	printk( "bmdrv: done in 2002 by sphaera & 1stein\n" );
	printk( "bmdrv: http://blinkenmini.1stein.no-ip.com/\n" );
	printk( "bmdrv: distributed under GNU Public License - no warranty\n" );
	printk( "bmdrv: trying to load (io=0x%X)...\n", io );

	//check startup parameters - io-address
	for( I = 0; I < count( BmDrvIoAddrs ); I++ )
		if( BmDrvIoAddrs[I] == (unsigned short)io )
			break;
	//invalid io-address
	if( I >= count( BmDrvIoAddrs ) )
	{
		//error message
		printk( "bmdrv: io=0x%X is not valid\n", io );
		printk( "bmdrv: valid settings for io are:" );
		for( I = 0; I < count( BmDrvIoAddrs ); I++ )
			printk( " 0x%X", BmDrvIoAddrs[I] );
		printk( "\n" );
		//init failed
		return -1;
	}
	//remember used io-address
	BmDrvIoAddr = BmDrvIoAddrs[I];

	//output device is not opend by a process yet
	BmDrvOutIsOpen = 0;
	BmDrvOutFile = NULL;

	//register device
	I = register_chrdev( BmDrvDevNoMajor, BmDrvDevName, &BmDrvFileOperations );
	//an error occured
	if( I < 0 )
	{
		//error-message
		printk( "bmdrv: register_chrdev failed with %d\n", I );
		//init failed
		return -1;
	}

	//register io-ports
	ResPtr = request_region( BmDrvIoAddr, 3, BmDrvDevName );
	//not free
	if( ResPtr == NULL )
	{
		//error-message
		printk( "bmdrv: request_region failed: io-ports not free\n" );
		//unregister device
		unregister_chrdev( BmDrvDevNoMajor, BmDrvDevName );
		//init failed
		return -1;
	}

	//setup hardware
	outb( 0x00, BmDrvIoAddr ); //set data-bits to 0
	outb( 0x03, BmDrvIoAddr + 2 ); //bidirectional: off, irq: off, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=0

	//set the picture to the standard picture
	memcpy( &BmDrvPicture, &BmDrvStdPicture, sizeof( struct tBmDrvInternalPicture ) );
	//output frame
	BmDrvFrameOut( );

	//init succeeded
	printk( "bmdrv: loaded (io=0x%X)...\n", BmDrvIoAddr );
	return 0;
}

//module is being unloaded
void cleanup_module( )
{
	//empty picture
	memset( &BmDrvPicture, 0, sizeof( struct tBmDrvInternalPicture ) );
	//output frame
	BmDrvFrameOut( );

	//reset hardware
	outb( 0x00, BmDrvIoAddr ); //set data-bits to 0
	outb( 0x0B, BmDrvIoAddr + 2 ); //bidirectional: off, irq: off, -nSTROBE=1, -nAUTOLF=1, nINIT=0, -nSLCTIN=1

	//unregister io-ports
	release_region( BmDrvIoAddr, 3 );

  	//unregister device
	unregister_chrdev( BmDrvDevNoMajor, BmDrvDevName );

	//done
	printk( "bmdrv: unloaded...\n" );
}

