/* bmgames.c
 * BLINKENmini games
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/time.h>
#include <errno.h>

#include "life.h"
#include "pong.h"
#include "snake.h"
#include "tetris.h"
#include "tron.h"

#define VerMaj 1
#define VerMin 1
#define VerRev 0

#define DisplaySizeXMin 8
#define DisplaySizeXDef 18
#define DisplaySizeXMax 128
#define DisplaySizeYMin 4
#define DisplaySizeYDef 8
#define DisplaySizeYMax 64

#define count( array ) (sizeof( (array) ) / sizeof( (array)[0] ))

//names of buttons
char * ButtonNames[] = { "B", "Y", "Select", "Start", "Up", "Down", "Left", "Right", "A", "X", "L", "R" };

//table with game information
typedef struct tGameInfo
{
	char * Name; //name of game
	void (* Init)( ); //init-procedure: called on game start
	void (* Exit)( ); //exit-procedure: called on game end
	void (* Tick)( ); //tick-procedure: called every 0.1 seconds
	void (* Action)( int Ctr, int Btn, int Press ); //action-procedure: called when button is pressed or released	
} stGameInfo;
struct tGameInfo GameInfoTable[] =
{
	{ "life", LifeInit, LifeExit, LifeTick, LifeAction },
	{ "pong", PongInit, PongExit, PongTick, PongAction },
	{ "snake", SnakeInit, SnakeExit, SnakeTick, SnakeAction },
	{ "tetris", TetrisInit, TetrisExit, TetrisTick, TetrisAction },
	{ "tron", TronInit, TronExit, TronTick, TronAction },
};

//current picture
int PictureSizeX, PictureSizeY;
unsigned char * * ppPicture;

//end variable
int End;

//signal handler to set end variable
void EndSignal( int SigNo )
{
	End = 1;
}

//controller button was pressed or released
void ControllerAction( int Controller, int Button, int Pressed )
{
	//display message
	if( Button < count( ButtonNames ) )
		if( Pressed )
			printf( "bmgames: button %s of controller %d was pressed\n", ButtonNames[Button], Controller );
		else
			printf( "bmgames: button %s of controller %d was released\n", ButtonNames[Button], Controller );
	else
		if( Pressed )
			printf( "bmgames: button %d of controller %d was pressed\n", Button, Controller );
		else
			printf( "bmgames: button %d of controller %d was released\n", Button, Controller );
}

//main
int main( int ArgC, char * * ArgV )
{
	int BmdrvOut, BmdrvIn, ReadBufferCnt = 0, I, Y;
	struct tGameInfo * pGameInfo;
	unsigned char ReadBuffer[4096], * pFrameBuffer, * pChr;

	//print message
	printf( "\n" );
	printf( "bmgames: BLINKENmini games (version %d.%d.%d)\n", VerMaj, VerMin, VerRev );
	printf( "         done in 2002 by sphaera & 1stein\n" );
	printf( "         http://blinkenmini.1stein.no-ip.com/\n" );
	printf( "         distributed under GNU Public License - no warranty\n\n" );

	//check arguments
	if( ArgC < 4 || ArgC > 5 )
	{
		printf( "%s <out-bmdrv> <in-bmdrv> <game> [<display-x>x<display-y>]\n\n", ArgV[0] );
		printf( "bmgames: supported games:" );
		for( I = 0; I < count( GameInfoTable ); I++ )
			printf( " %s", GameInfoTable[I].Name );
		printf( "\n\n" );
		printf( "bmgames: supported display sizes: %d..%dx%d..%d (%dx%d is default)\n\n", DisplaySizeXMin, DisplaySizeXMax, DisplaySizeYMin, DisplaySizeYMax, DisplaySizeXDef, DisplaySizeYDef );
		return 1;
	}

	//get game
	for( I = 0; I < count( GameInfoTable ); I++ )
		if( strcasecmp( GameInfoTable[I].Name, ArgV[3] ) == 0 )
			break;
	if( I < count( GameInfoTable ) )
		pGameInfo = &GameInfoTable[I];
	else
	{
		printf( "bmgames: unsupported game: %s\n\n", ArgV[3] );
		printf( "bmgames: supported games:" );
		for( I = 0; I < count( GameInfoTable ); I++ )
			printf( " %s", GameInfoTable[I].Name );
		printf( "\n\n" );
		return 1;
	}
	//print game
	printf( "bmgames: selected game is %s\n\n", pGameInfo->Name );

	//get display size
	PictureSizeX = DisplaySizeXDef;
	PictureSizeY = DisplaySizeYDef;
	if( ArgC == 5 )
	{
		//parse argument
		PictureSizeX = atol( ArgV[4] );
		pChr = strchr( ArgV[4], 'x' );
		if( pChr == NULL )
			pChr = strchr( ArgV[4], 'X' );
		if( pChr == NULL )
			PictureSizeY = 0;
		else
			PictureSizeY = atol( pChr + 1 );
		//check display size
		if( PictureSizeX < DisplaySizeXMin || PictureSizeX > DisplaySizeXMax
			|| PictureSizeY < DisplaySizeYMin || PictureSizeY > DisplaySizeYMax )
		{
			printf( "bmgames: invalid display size: %dx%d\n\n", PictureSizeX, PictureSizeY );
			printf( "bmgames: supported display sizes: %d..%dx%d..%d (%dx%d is default)\n\n", DisplaySizeXMin, DisplaySizeXMax, DisplaySizeYMin, DisplaySizeYMax, DisplaySizeXDef, DisplaySizeYDef );
			return -1;
		}
	}

	//open bmdrv
	BmdrvOut = open( ArgV[1], O_WRONLY | O_NOCTTY | O_NONBLOCK );
	if( BmdrvOut == -1 )
	{
		printf( "bmgames: could not open %s: %s\n\n", ArgV[1], strerror( errno ) );
		return -1;
	}
	BmdrvIn = open( ArgV[2], O_RDONLY | O_NOCTTY | O_NONBLOCK );
	if( BmdrvIn == -1 )
	{
		printf( "bmgames: could not open %s: %s\n\n", ArgV[2], strerror( errno ) );
		close( BmdrvOut );
		return -1;
	}

	//allocate buffer for picture
	ppPicture = (unsigned char * *)malloc( PictureSizeY * sizeof( unsigned char * ) );
	for( Y = 0; Y < PictureSizeY; Y++ )
		ppPicture[Y] = (unsigned char *)malloc( PictureSizeX );
	//clear current picture
	for( Y = 0; Y < PictureSizeY; Y++ )
		memset( ppPicture[Y], 0, PictureSizeX );

	//allocate frame buffer
	pFrameBuffer = (unsigned char *)malloc( PictureSizeY * PictureSizeX + 3 );

	//initialize random number generator
	srand( time( NULL ) );

	//install signal handlers
	signal( SIGHUP, EndSignal );
	signal( SIGINT, EndSignal );
	signal( SIGQUIT, EndSignal );

	//call init-procedure
	if( pGameInfo->Init != NULL )
		pGameInfo->Init( );

	for( End = 0; End == 0; )
	{
		//read from bmdrv
		I = read( BmdrvIn, &ReadBuffer + ReadBufferCnt, sizeof( ReadBuffer ) - ReadBufferCnt );
		if( I == -1 && errno != EAGAIN )
		{
			printf( "bmgames: could not read from %s: %s\n\n", ArgV[2], strerror( errno ) );
			if( pGameInfo->Exit != NULL )
				pGameInfo->Exit( );
			free( pFrameBuffer );
			for( Y = 0; Y < PictureSizeY; Y++ )
				free( ppPicture[Y] );
			free( ppPicture );
			close( BmdrvIn );
			close( BmdrvOut );
			return -1;
		}
		if( I > 0 )
			ReadBufferCnt += I;

		//process controller button actions
		for( I = 0; I < ReadBufferCnt && ReadBuffer[I] != 0xFF; I++ ); //ignore everything before 0xFF
		for( ; I + 3 < ReadBufferCnt; I += 4 ) //as long, as complete actions are availible
		{
			//process this action
			ControllerAction( ReadBuffer[I + 2], ReadBuffer[I + 3], ReadBuffer[I + 1] );
			//call action-procedure
			if( pGameInfo->Action != NULL )
				pGameInfo->Action( ReadBuffer[I + 2], ReadBuffer[I + 3], ReadBuffer[I + 1] );
		}
		ReadBufferCnt -= I; //remove processed actions from buffer
		memmove( ReadBuffer, ReadBuffer + I, ReadBufferCnt );

		//copy current picture to frame buffer
		pFrameBuffer[0] = (unsigned char)PictureSizeX;
		pFrameBuffer[1] = (unsigned char)PictureSizeY;
		for( Y = 0; Y < PictureSizeY; Y++ )
			memcpy( pFrameBuffer + 2 + Y * PictureSizeX, ppPicture[Y], PictureSizeX );
		pFrameBuffer[PictureSizeY * PictureSizeX + 2] = 0;

		//write frame to bmdrv
		I = write( BmdrvOut, pFrameBuffer, PictureSizeY * PictureSizeX + 3 );
		if( I != PictureSizeY * PictureSizeX + 3 )
		{
			printf( "bmgames: could not write to %s: %s\n\n", ArgV[1], strerror( errno ) );
			if( pGameInfo->Exit != NULL )
				pGameInfo->Exit( );
			free( pFrameBuffer );
			for( Y = 0; Y < PictureSizeY; Y++ )
				free( ppPicture[Y] );
			free( ppPicture );
			close( BmdrvIn );
			close( BmdrvOut );
			return -1;
		}

		//call tick-procedure
		if( pGameInfo->Tick != NULL )
			pGameInfo->Tick( );

		//wait 0.1 seconds
		usleep( 100000 );
	} //for( End = 0; End == 0; )

	//call exit-procedure
	if( pGameInfo->Exit != NULL )
		pGameInfo->Exit( );

	//free frame buffer
	free( pFrameBuffer );

	//free picture
	for( Y = 0; Y < PictureSizeY; Y++ )
		free( ppPicture[Y] );
	free( ppPicture );

	//close bmdrv
	close( BmdrvIn );
	close( BmdrvOut );

	printf( "\nbmgames: done...\n\n" );
	return 0;
}

