/* snake.c
 * BLINKENmini games
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>

#include "keys.h"
#include "snake.h"

#define SnakeLengthMaxMax 256
#define SnakeMouseCntMaxMax 16

#define SnakeHeadColor 0xFF
#define SnakeHeadBlinkColor 0xBF
#define SnakeBodyColor 0xBF
#define SnakeBodyBlinkColor 0x7F
#define SnakeMouseColor 0xFF
#define SnakeEmptyColor 0x00

#define count( array ) (sizeof( (array) ) / sizeof( (array)[0] ))

//current picture
extern int PictureSizeX, PictureSizeY;
extern unsigned char * * ppPicture;

//global variables
int SnakeTickCnt; //tick counter
int SnakeDead; //is set to something different from 0 if snake is dead (and is decremented then)
int SnakeLength; //current length of snake
int SnakeLengthMax; //maximum langth of snake
int SnakePosX[SnakeLengthMaxMax], SnakePosY[SnakeLengthMaxMax]; //position of snake segments, 0 is head
int SnakeDirX, SnakeDirY; //direction, snake is moving in
int SnakeMouseCnt; //numer of currently availible mice
int SnakeMouseCntMax; //maximum number of mice
int SnakeMouseX[SnakeMouseCntMaxMax], SnakeMouseY[SnakeMouseCntMaxMax]; //position of mice

//output current picture
void SnakeOutput( )
{
	int X, Y, I, BodyColor, HeadColor;

	//get colors
	if( SnakeDead & 1 )
	{
		HeadColor = SnakeHeadBlinkColor;
		BodyColor = SnakeBodyBlinkColor;
	}
	else
	{
		HeadColor = SnakeHeadColor;
		BodyColor = SnakeBodyColor;
	}

	//empty picture
	for( Y = 0; Y < PictureSizeY; Y++ )
		for( X = 0; X < PictureSizeX; X++ )
			ppPicture[Y][X] = SnakeEmptyColor;

	//put mice into picture
	for( I = 0; I < SnakeMouseCnt; I++ )
		ppPicture[SnakeMouseY[I]][SnakeMouseX[I]] = SnakeMouseColor;

	//put snake body into picture
	for( I = 1; I < SnakeLength; I++ )
		ppPicture[SnakePosY[I]][SnakePosX[I]] = BodyColor;
	//put snake head into picture
	ppPicture[SnakePosY[0]][SnakePosX[0]] = HeadColor;
}

//create a mouse
void SnakeNewMouse( )
{
	int MouseX, MouseY, I;

	//max. mouse count reached
	if( SnakeMouseCnt >= SnakeMouseCntMax )
		return;

	//ask random numer generator if we need a mouse
	if( SnakeMouseCnt == 0 )
	{
		//no mouse on the field - high probability
		if( rand( ) % 100 > 10 )
			return;
	}
	else
	{
		//there is a mouse on the field - low probability
		if( rand( ) % 100 > 1 )
			return;
	}

	//get a position
	MouseX = rand( ) % PictureSizeX;
	MouseY = rand( ) % PictureSizeY;

	//check if position is free
	for( I = 0; I < SnakeLength; I++ )
		if( SnakePosX[I] == MouseX && SnakePosY[I] == MouseY )
			return;
	for( I = 0; I < SnakeMouseCnt; I++ )
		if( SnakeMouseX[I] == MouseX && SnakeMouseY[I] == MouseY )
			return;

	//put mouse at this position
	SnakeMouseX[SnakeMouseCnt] = MouseX;
	SnakeMouseY[SnakeMouseCnt] = MouseY;
	SnakeMouseCnt++;

	//output new picture
	SnakeOutput( );
}

//start new game
void SnakeNewGame( )
{
	int I;

	//set snake head to center
	SnakePosX[0] = (PictureSizeX - 1) / 2 + rand( ) % 3 - 1;
	SnakePosY[0] = (PictureSizeY - 1) / 2 + rand( ) % 3 - 1;

	//set snake direction
	switch( rand( ) % 4 )
	{
		case 0: SnakeDirX = -1; SnakeDirY =  0; break;
		case 1: SnakeDirX =  1; SnakeDirY =  0; break;
		case 2: SnakeDirX =  0; SnakeDirY = -1; break;
		case 3: SnakeDirX =  0; SnakeDirY =  1; break;
	}

	//set snake length to 3
	SnakeLength = 3;

	//set up body of snake
	for( I = 1; I < SnakeLength; I++ )
	{
		SnakePosX[I] = SnakePosX[0] - I * SnakeDirX;
		SnakePosY[I] = SnakePosY[0] - I * SnakeDirY;
	}

	//snake is not dead
	SnakeDead = 0;

	//no mice
	SnakeMouseCnt = 0;

	//print message
	printf( "bmgames: snake: new game started\n" );

	//output new picture
	SnakeOutput( );
}

//init-procedure: called at game start
void SnakeInit( )
{
	//default values for variables
	SnakeTickCnt = 0;
	//calculate max values for snake length and mouse count
	SnakeLengthMax = PictureSizeX + PictureSizeY;
	if( SnakeLengthMax > SnakeLengthMaxMax )
		SnakeLengthMax = SnakeLengthMaxMax;
	SnakeMouseCntMax = 3;
	if( SnakeMouseCntMax > SnakeMouseCntMaxMax )
		SnakeMouseCntMax = SnakeMouseCntMaxMax;

	//print message
	printf( "bmgames: snake: init\n" );

	//new game
	SnakeNewGame( );
}

//exit-procedure: called on game end
void SnakeExit( )
{
	//print message
	printf( "bmgames: snake: exit\n" );
}

//tick-procedure: called every 0.1 seconds
void SnakeTick( )
{
	int HeadX, HeadY, I, J, Crash;

	//only do something every 0.5 seconds
	SnakeTickCnt++;
	if( SnakeTickCnt < 5 )
	{
		//create a mouse
		SnakeNewMouse( );
		return;
	}
	SnakeTickCnt = 0;

	//snake is dead
	if( SnakeDead )
	{
		//decrement dead-counter
		SnakeDead--;
		//start new game
		if( SnakeDead == 0 )
			SnakeNewGame( );
		//output new picture
		else
			SnakeOutput( );
		//do nothing else
		return;
	}

	//get new position of head
	HeadX = SnakePosX[0] + SnakeDirX;
	HeadY = SnakePosY[0] + SnakeDirY;

	//check if snake crashed into wall
	if( HeadX < 0 || HeadX >= PictureSizeX || HeadY < 0 || HeadY >= PictureSizeY )
	{
		//print message
		printf( "bmgames: snake: snake crashed into wall\n" );
		//snake is dead
		SnakeDead = 5;
		//output new picture
		SnakeOutput( );
		//do nothing else
		return;
	}
	//snake crashed into its body (last field is not important, because snake moves one step)
	Crash = 0;
	for( I = 1; I < SnakeLength - 1; I++ )
		if( HeadX == SnakePosX[I] && HeadY == SnakePosY[I] )
			Crash = 1;
	if( Crash )
	{
		//print message
		printf( "bmgames: snake: snake crashed into its body\n" );
		//snake is dead
		SnakeDead = 5;
		//output new picture
		SnakeOutput( );
		//do nothing else
		return;
	}

	//check if snake ate a mouse
	for( I = 0; I < SnakeMouseCnt; I++ )
		if( SnakeMouseX[I] == HeadX && SnakeMouseY[I] == HeadY )
			break;
	//snake ate a mouse
	if( I < SnakeMouseCnt )
	{
		//increase snake length
		if( SnakeLength < SnakeLengthMaxMax )
			SnakeLength++;
		//remove mouse
		SnakeMouseCnt--;
		for( J = I; J < SnakeMouseCnt; J++ )
		{
			SnakeMouseX[J] = SnakeMouseX[J + 1];
			SnakeMouseY[J] = SnakeMouseY[J + 1];
		}
		//print message
		printf( "bmgames: snake ate a mouse and is now %d pixels long\n", SnakeLength );
	}

	//move snake
	for( I = SnakeLength - 1; I > 0; I-- )
	{
		SnakePosX[I] = SnakePosX[I - 1];
		SnakePosY[I] = SnakePosY[I - 1];
	}
	SnakePosX[0] = HeadX;
	SnakePosY[0] = HeadY;

        //output new picture
	SnakeOutput( );
}

//action-procedure: called when button is pressed or released	
void SnakeAction( int Ctr, int Btn, int Press )
{
	//only process pressed buttons of controller 0
	if( !Press || Ctr != 0 )
		return;

	//different buttons
	switch( Btn )
	{
		//left: move leftwards
		case KeyLeft:
			SnakeDirX = -1;
			SnakeDirY =  0;
			break;

		//right: move rightwards
		case KeyRight:
			SnakeDirX =  1;
			SnakeDirY =  0;
			break;

		//up: move upwards
		case KeyUp:
			SnakeDirX =  0;
			SnakeDirY = -1;
			break;

		//down: move downwards
		case KeyDown:
			SnakeDirX =  0;
			SnakeDirY =  1;
			break;
	}
}

