/* tetris.c
 * BLINKENmini games
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>

#include "keys.h"
#include "tetris.h"

#define TetrisActiveStoneColor 0xFF
#define TetrisSolidStoneColor 0xBF
#define TetrisEmptyColor 0x00

#define count( array ) (sizeof( (array) ) / sizeof( (array)[0] ))

//current picture
extern int PictureSizeX, PictureSizeY;
extern unsigned char * * ppPicture;

//types
typedef struct tTetrisPoint
{
	int X, Y;
} stTetrisPoint;
typedef struct tTetrisStone
{
	struct tTetrisPoint Pos[4]; //position of the 4 fields (seen from the center)
} stTetrisStone;

//stones (always 4 lines with same stone in different rotation angles)
struct tTetrisStone TetrisStoneTable[] =
{
	{ { { -2,  0 }, { -1,  0 }, {  0,  0 }, {  1,  0 } } }, //the I
	{ { {  0, -2 }, {  0, -1 }, {  0,  0 }, {  0,  1 } } },
	{ { { -2,  0 }, { -1,  0 }, {  0,  0 }, {  1,  0 } } },
	{ { {  0, -2 }, {  0, -1 }, {  0,  0 }, {  0,  1 } } },
	{ { {  1, -1 }, { -1,  0 }, {  0,  0 }, {  1,  0 } } }, //the L
	{ { {  0, -1 }, {  0,  0 }, {  0,  1 }, {  1,  1 } } },
	{ { { -1,  0 }, {  0,  0 }, {  1,  0 }, { -1,  1 } } },
	{ { { -1, -1 }, {  0, -1 }, {  0,  0 }, {  0,  1 } } },
	{ { { -1, -1 }, { -1,  0 }, {  0,  0 }, {  1,  0 } } }, //the J
	{ { {  0, -1 }, {  1, -1 }, {  0,  0 }, {  0,  1 } } },
	{ { { -1,  0 }, {  0,  0 }, {  1,  0 }, {  1,  1 } } },
	{ { {  0, -1 }, {  0,  0 }, {  -1, 1 }, {  0,  1 } } },
	{ { {  0, -1 }, { -1,  0 }, {  0,  0 }, {  1,  0 } } }, //the T
	{ { {  0, -1 }, {  0,  0 }, {  1,  0 }, {  0,  1 } } },
	{ { { -1,  0 }, {  0,  0 }, {  1,  0 }, {  0,  1 } } },
	{ { {  0, -1 }, { -1,  0 }, {  0,  0 }, {  0,  1 } } },
	{ { {  0, -1 }, {  1, -1 }, {  0,  0 }, {  1,  0 } } }, //the block
	{ { {  0, -1 }, {  1, -1 }, {  0,  0 }, {  1,  0 } } },
	{ { {  0, -1 }, {  1, -1 }, {  0,  0 }, {  1,  0 } } },
	{ { {  0, -1 }, {  1, -1 }, {  0,  0 }, {  1,  0 } } },
	{ { { -1, -1 }, {  0, -1 }, {  0,  0 }, {  1,  0 } } }, //the Z
	{ { {  0, -1 }, { -1,  0 }, {  0,  0 }, { -1,  1 } } },
	{ { { -1, -1 }, {  0, -1 }, {  0,  0 }, {  1,  0 } } },
	{ { {  0, -1 }, { -1,  0 }, {  0,  0 }, { -1,  1 } } },
	{ { {  0, -1 }, {  1, -1 }, { -1,  0 }, {  0,  0 } } }, //the S
	{ { { -1, -1 }, { -1,  0 }, {  0,  0 }, {  0,  1 } } },
	{ { {  0, -1 }, {  1, -1 }, { -1,  0 }, {  0,  0 } } },
	{ { { -1, -1 }, { -1,  0 }, {  0,  0 }, {  0,  1 } } },
};

//global variables
int TetrisTickCnt; //tick counter
unsigned char * * ppTetrisField; //pointer to game field
int TetrisActiveStoneIndex; //index of active stone, -1 if none
int TetrisActiveStonePosX, TetrisActiveStonePosY; //position of active stone

//check if stone would fit at a position
//returns boolean result
int TetrisStoneFit( int StoneIndex, int PosX, int PosY )
{
	int I, X, Y;

	//check all parts of stone
	for( I = 0; I < 4; I++ )
	{
		//get position of stone part
		X = PosX + TetrisStoneTable[StoneIndex].Pos[I].X;
		Y = PosY + TetrisStoneTable[StoneIndex].Pos[I].Y;
		//out of field (out of field at top is okay)
		if( X < 0 || X >= PictureSizeX || Y >= PictureSizeY )
			return 0;
		//within field
		if( Y >= 0 )
			//field not free
			if( ppTetrisField[Y][X] != 0 )
				return 0;
	}

	//stone fits
	return 1;
}

//output current picture
void TetrisOutput( )
{
	int X, Y, I;

	//copy game field to picture
	for( Y = 0; Y < PictureSizeY; Y++ )
		for( X = 0; X < PictureSizeX; X++ )
			if( ppTetrisField[Y][X] )
				ppPicture[Y][X] = TetrisSolidStoneColor;
			else
				ppPicture[Y][X] = TetrisEmptyColor;

	//put active stone into picture
	if( TetrisActiveStoneIndex >= 0 )
	{
		for( I = 0; I < 4; I++ )
		{
			X = TetrisActiveStonePosX + TetrisStoneTable[TetrisActiveStoneIndex].Pos[I].X;
			Y = TetrisActiveStonePosY + TetrisStoneTable[TetrisActiveStoneIndex].Pos[I].Y;
			if( X >= 0 && X < PictureSizeX && Y >= 0 && Y < PictureSizeY )
				ppPicture[Y][X] = TetrisActiveStoneColor;
		}
	}	
}

//start new game
void TetrisNewGame( )
{
	int X, Y;

	//no active stone
	TetrisActiveStoneIndex = -1;

	//clear game field
	for( Y = 0; Y < PictureSizeY; Y++ )
		for( X = 0; X < PictureSizeX; X++ )
			ppTetrisField[Y][X] = 0;

	//print message
	printf( "bmgames: tetris: new game started\n" );

	//output picture
	TetrisOutput( );
}

//init-procedure: called at game start
void TetrisInit( )
{
	int Y;

	//set global variables
	TetrisTickCnt = 0;

	//allocate game field
	ppTetrisField = (unsigned char * *)malloc( PictureSizeY * sizeof( unsigned char * ) );
	for( Y = 0; Y < PictureSizeY; Y++ )
		ppTetrisField[Y] = (unsigned char *)malloc( PictureSizeX );

	//print message
	printf( "bmgames: tetris: init\n" );

	//start new game
	TetrisNewGame( );
}

//exit-procedure: called on game end
void TetrisExit( )
{
	int Y;

	//free game field
	for( Y = 0; Y < PictureSizeY; Y++ )
		free( ppTetrisField[Y] );
	free( ppTetrisField );

	//print message
	printf( "bmgames: tetris: exit\n" );
}

//tick-procedure: called every 0.1 seconds
void TetrisTick( )
{
	int I, X, Y, YY;

	//only do something every second
	TetrisTickCnt++;
	if( TetrisTickCnt < 10 )
		return;
	TetrisTickCnt = 0;

	//no active stone
	if( TetrisActiveStoneIndex < 0 )
	{
		//remove the first complete line
		for( Y = 0; Y < PictureSizeY; Y++ )
		{
			//check if line is complete
			for( X = 0; X < PictureSizeX; X++ )
				if( !ppTetrisField[Y][X] )
					break;
			//line is complete
			if( X >= PictureSizeX )
			{
				//print message
				printf( "bmgames: tetris: line %d was complete and is removed\n", Y + 1 );
				//move all lines above this line down one field
				for( YY = Y; YY > 0; YY-- )
					for( X = 0; X < PictureSizeX; X++ )
						ppTetrisField[YY][X] = ppTetrisField[YY - 1][X];
				//clear first line
				for( X = 0; X < PictureSizeX; X++ )
					ppTetrisField[0][X] = 0;
				//end loop
				break;
			}
		}

		//no complete line found
		if( Y >= PictureSizeY )
		{
			//print message
			printf( "bmgames: tetris: new stone\n" );
			//select a new stone
			TetrisActiveStoneIndex = rand( ) % count( TetrisStoneTable );
			//put stone at center of the top of the field
			TetrisActiveStonePosX = (PictureSizeX - 1) / 2;
			TetrisActiveStonePosY = -2;
		}
	}
	//a stone is active
	else
	{
		//stone can fall one down
		if( TetrisStoneFit( TetrisActiveStoneIndex, TetrisActiveStonePosX, TetrisActiveStonePosY + 1 ) )
		{
			//stone falls one down
			TetrisActiveStonePosY++;
		}
		//stone cannot fall one down
		else
		{
			//print message
			printf( "bmgames: tetris: stone hit ground\n" );
			//stone becomes solid
			for( I = 0; I < 4; I++ )
			{
				X = TetrisActiveStonePosX + TetrisStoneTable[TetrisActiveStoneIndex].Pos[I].X;
				Y = TetrisActiveStonePosY + TetrisStoneTable[TetrisActiveStoneIndex].Pos[I].Y;
				if( X >= 0 && X < PictureSizeX && Y >= 0 && Y < PictureSizeY )
					ppTetrisField[Y][X] = 1;
			}
			//no active stone now
			TetrisActiveStoneIndex = -1;

			//game over if stone was still above field when it hit
			if( TetrisActiveStonePosY < 0 )
			{
				//start new game
				TetrisNewGame( );
				//do nothing else
				return;
			}
		}
	}

        //output new picture
	TetrisOutput( );
}

//action-procedure: called when button is pressed or released	
void TetrisAction( int Ctr, int Btn, int Press )
{
	int I;

	//only process pressed buttons of controller 0
	if( !Press || Ctr != 0 )
		return;

	//different buttons
	switch( Btn )
	{
		//left: move stone left
		case KeyLeft:
			//move stone 1 left
			if( TetrisStoneFit( TetrisActiveStoneIndex, TetrisActiveStonePosX - 1, TetrisActiveStonePosY ) )
				TetrisActiveStonePosX--;
			//output new picture
			TetrisOutput( );
			break;

		//right: move stone right
		case KeyRight:
			//move stone 1 right
			if( TetrisStoneFit( TetrisActiveStoneIndex, TetrisActiveStonePosX + 1, TetrisActiveStonePosY ) )
				TetrisActiveStonePosX++;
			//output new picture
			TetrisOutput( );
			break;

		//down: drop stone
		case KeyDown:
			//move stone down until it hits
			while( TetrisStoneFit( TetrisActiveStoneIndex, TetrisActiveStonePosX, TetrisActiveStonePosY + 1 ) )
				TetrisActiveStonePosY++;
			//output new picture
			TetrisOutput( );
			break;

		//B: turn stone clockwise
		case KeyB:
			//turn stone 1 time
			I = (TetrisActiveStoneIndex & ~3) | (((TetrisActiveStoneIndex & 3) + 1) & 3);
			if( TetrisStoneFit( I, TetrisActiveStonePosX, TetrisActiveStonePosY ) )
				TetrisActiveStoneIndex = I;
			//output new picture
			TetrisOutput( );
			break;

		//Y: turn stone counter-clockwise
		case KeyY:
			//turn stone 3 times
			I = (TetrisActiveStoneIndex & ~3) | (((TetrisActiveStoneIndex & 3) + 3) & 3);
			if( TetrisStoneFit( I, TetrisActiveStonePosX, TetrisActiveStonePosY ) )
				TetrisActiveStoneIndex = I;
			//output new picture
			TetrisOutput( );
			break;
	}
}

