/* tron.c
 * BLINKENmini games
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>

#include "keys.h"
#include "tron.h"

#define TronPlayerCnt 4
#define TronPlayerColor 0xFF
#define TronPlayerNotReadyColor 0x9F
#define TronWallColor 0xBF
#define TronEmptyColor 0x00

#define count( array ) (sizeof( (array) ) / sizeof( (array)[0] ))

//current picture
extern int PictureSizeX, PictureSizeY;
extern unsigned char * * ppPicture;

//global variables
int TronTickCnt; //tick counter
int TronRunning; //if game is already running
int TronGameOver; //if running game is over
unsigned char * * ppTronField; //pointer to game field
int TronPlayerActive[TronPlayerCnt]; //if players are active
int TronPlayerReady[TronPlayerCnt]; //if player is ready (always true for active players when game is running)
int TronPlayerPosX[TronPlayerCnt], TronPlayerPosY[TronPlayerCnt]; //position of players
int TronPlayerDirX[TronPlayerCnt], TronPlayerDirY[TronPlayerCnt]; //directions of players

//output current picture
void TronOutput( )
{
	int X, Y, I;

	//copy game field to picture
	for( Y = 0; Y < PictureSizeY; Y++ )
		for( X = 0; X < PictureSizeX; X++ )
			if( ppTronField[Y][X] )
				ppPicture[Y][X] = TronWallColor;
			else
				ppPicture[Y][X] = TronEmptyColor;

	//put players into picture
	for( I = 0; I < TronPlayerCnt; I++ )
	{
		if( TronPlayerActive[I] )
		{
			if( TronPlayerReady[I] )
				ppPicture[TronPlayerPosY[I]][TronPlayerPosX[I]] = TronPlayerColor;
			else
				ppPicture[TronPlayerPosY[I]][TronPlayerPosX[I]] = TronPlayerNotReadyColor;
		}
	}
}

//start new game
void TronNewGame( )
{
	int X, Y, I;

	//clear game field
	for( Y = 0; Y < PictureSizeY; Y++ )
		for( X = 0; X < PictureSizeX; X++ )
			ppTronField[Y][X] = 0;

	//game not runnig, players are joining
	TronRunning = 0;
	TronGameOver = 0;

	//no player is active or ready
	for( I = 0; I < TronPlayerCnt; I++ )
	{
		TronPlayerActive[I] = 0;
		TronPlayerReady[I] = 0;
	}

	//print message
	printf( "bmgames: tron: new game opened\n" );

	//output picture
	TronOutput( );
}

//init-procedure: called at game start
void TronInit( )
{
	int Y;

	//allocate game field
	ppTronField = (unsigned char * *)malloc( PictureSizeY * sizeof( unsigned char * ) );
	for( Y = 0; Y < PictureSizeY; Y++ )
		ppTronField[Y] = (unsigned char *)malloc( PictureSizeX );

	//print message
	printf( "bmgames: tron: init\n" );

	//start new game
	TronNewGame( );
}

//exit-procedure: called on game end
void TronExit( )
{
	int Y;

	//free game field
	for( Y = 0; Y < PictureSizeY; Y++ )
		free( ppTronField[Y] );
	free( ppTronField );

	//print message
	printf( "bmgames: tron: exit\n" );
}

//tick-procedure: called every 0.1 seconds
void TronTick( )
{
	int I, Crash, PlrCnt;

	//do nothing, if game is not yet running or is already over
	if( !TronRunning || TronGameOver )
		return;

	//only do something every 0.3 seconds
	TronTickCnt++;
	if( TronTickCnt < 3 )
		return;
	TronTickCnt = 0;

	//advance players
	for( I = 0; I < TronPlayerCnt; I++ )
	{
		if( TronPlayerActive[I] )
		{
			//put a wall at old player position
			ppTronField[TronPlayerPosY[I]][TronPlayerPosX[I]] = 1;
			//move player one field in its direction (wrap around borders of field)
			TronPlayerPosX[I] += TronPlayerDirX[I];
			if( TronPlayerPosX[I] < 0 )
				TronPlayerPosX[I] = PictureSizeX - 1;
			if( TronPlayerPosX[I] >= PictureSizeX )
				TronPlayerPosX[I] = 0;
			TronPlayerPosY[I] += TronPlayerDirY[I];
			if( TronPlayerPosY[I] < 0 )
				TronPlayerPosY[I] = PictureSizeY - 1;
			if( TronPlayerPosY[I] >= PictureSizeY )
				TronPlayerPosY[I] = 0;
		}
	}

	//check if some player chrashed
	Crash = 0;
	for( I = 0; I < TronPlayerCnt; I++ )
	{
		if( TronPlayerActive[I] )
		{
			//player chrashed
			if( ppTronField[TronPlayerPosY[I]][TronPlayerPosX[I]] )
			{
				//player is now inactive and (not ready)
				TronPlayerActive[I] = 0;

				//print message
				if( !Crash )
					printf( "bmgames: tron: player(s)" );
				Crash = 1;
				printf( " %d", I );
			}
		}
	}
	//print rest of message
	if( Crash )
		printf( " crashed\n" );

	//count players
	PlrCnt = 0;
	for( I = 0; I < TronPlayerCnt; I++ )
		if( TronPlayerActive[I] )
			PlrCnt++;
	//less than two players
	if( PlrCnt < 2 )
	{
		//find last player
		for( I = 0; I < TronPlayerCnt; I++ )
			if( TronPlayerActive[I] )
				break;
		//print message
		if( I < TronPlayerCnt )
			printf( "bmgames: tron: player %d won the game\n", I );
		else
			printf( "bmgames: tron: no player won the game\n" );

		//game over
		TronGameOver = 1;
	}	

        //output new picture
	TronOutput( );
}

//action-procedure: called when button is pressed or released	
void TronAction( int Ctr, int Btn, int Press )
{
	int I, X, Y, Free, PlrCnt, PlrRdyCnt;

	//only process pressed buttons of controllers 0..(PlayerCnt-1)
	if( !Press || Ctr >= TronPlayerCnt )
		return;

	//different buttons
	switch( Btn )
	{
		//select: join game / close finished game
		case KeySelect:
			//player inactive, game not running
			if( !TronPlayerActive[Ctr] && !TronRunning )
			{
				//randomize players position
				do
				{
					//get a position somewhere in the middle of the field
					TronPlayerPosX[Ctr] = rand( ) % (PictureSizeX - 4) + 2;
					TronPlayerPosY[Ctr] = rand( ) % (PictureSizeY - 4) + 2;
					//check if position is free
					Free = 1;
					for( I = 0; I < TronPlayerCnt; I++ )
						if( TronPlayerActive[I] )
							for( Y = TronPlayerPosY[Ctr] - 2; Y <= TronPlayerPosY[Ctr] + 2; Y++ )
								for( X = TronPlayerPosX[Ctr] - 2; X <= TronPlayerPosX[Ctr] + 2; X++ )
									if( X == TronPlayerPosX[I] && Y == TronPlayerPosY[I] )
										Free = 0;
				}
				while( !Free ); //until position is free
				//randomize players direction
				switch( rand( ) % 4 )
				{
					case 0: TronPlayerDirX[Ctr] = -1; TronPlayerDirY[Ctr] =  0; break;
					case 1: TronPlayerDirX[Ctr] =  1; TronPlayerDirY[Ctr] =  0; break;
					case 2: TronPlayerDirX[Ctr] =  0; TronPlayerDirY[Ctr] = -1; break;
					case 3: TronPlayerDirX[Ctr] =  0; TronPlayerDirY[Ctr] =  1; break;
				}
				//set player to not yet ready
				TronPlayerReady[Ctr] = 0;
				//join game
				TronPlayerActive[Ctr] = 1;

				//print message
				printf( "bmgames: tron: player %d joined the game\n", Ctr );

				//output new picture
				TronOutput( );
				break;
			}
			//game running and already over
			if( TronRunning && TronGameOver )
			{
				//open a new game
				TronNewGame( );
				break;
			}
			break;

		//start: begin game
		case KeyStart:
			//player active, game not running
			if( TronPlayerActive[Ctr] && !TronRunning )
			{
				//toggle ready-status
				TronPlayerReady[Ctr] = !TronPlayerReady[Ctr];

				//print message
				if( TronPlayerReady[Ctr] )
					printf( "bmgames: tron: player %d is ready now\n", Ctr );
				else
					printf( "bmgames: tron: player %d is not ready again\n", Ctr );

				//output new picture
				TronOutput( );
				
				//count active players and active and ready players
				PlrCnt = PlrRdyCnt = 0;
				for( I = 0; I < TronPlayerCnt; I++ )
				{
					if( TronPlayerActive[I] )
					{
						PlrCnt++;
						if( TronPlayerReady[I] )
							PlrRdyCnt++;
					}
				}
				//at least two players there and all players ready
				if( PlrCnt >= 2 && PlrRdyCnt == PlrCnt )
				{
					//start game
					TronRunning = 1;
					TronGameOver = 0;
					TronTickCnt = 0;

					//print message
					printf( "bmgames: tron: game started\n" );
				}
				break;
			}
			break;

		//left: set direction to leftwards
		case KeyLeft:
			TronPlayerDirX[Ctr] = -1;
			TronPlayerDirY[Ctr] =  0;
			break;

		//right: set direction to rightwards
		case KeyRight:
			TronPlayerDirX[Ctr] =  1;
			TronPlayerDirY[Ctr] =  0;
			break;

		//up: set direction to upwards
		case KeyUp:
			TronPlayerDirX[Ctr] =  0;
			TronPlayerDirY[Ctr] = -1;
			break;

		//down: set direction to downwards
		case KeyDown:
			TronPlayerDirX[Ctr] =  0;
			TronPlayerDirY[Ctr] =  1;
			break;
	}
}

