/* bmsim.c
 * BLINKENmini simulator
 *  - simulates the character device bmdrv on a console
 * Copyright (C) 2002 sphaera & 1stein (http://blinkenmini.1stein.no-ip.com/)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <errno.h>

#define VerMaj 1
#define VerMin 0
#define VerRev 0

#define KeyStatePressVal 5
#define WaitTime 20000

#define DisplaySizeX 18
#define DisplaySizeY 8

#define LineNoPicture 6
#define LineNoKeys (LineNoPicture + DisplaySizeY + 1)
#define LineNoControllers (LineNoKeys + 1)
#define LineNoEnd (LineNoControllers + 8)

#define count( array ) (sizeof( (array) ) / sizeof( (array)[0] ))

//keys for controllers
#define ControllerCnt 2
#define KeyCnt 12
#define KeyB 0
#define KeyY 1
#define KeySelect 2
#define KeyStart 3
#define KeyUp 4
#define KeyDown 5
#define KeyLeft 6
#define KeyRight 7
#define KeyA 8
#define KeyX 9
#define KeyL 10
#define KeyR 11
struct tControllerKeys
{
	char * Keys[ControllerCnt]; //each string must be KeyCnt characters long and contain the keys in above order
} ControllerKeys[] =
{
	{ { "r4xcwsadt5  ", "i8nmujhko9  " } },
	{ { "trxcwsad65  ", "oinmujhk09  " } },
};

//current controller key layout
int ControllerKeyIndex;

//current key state
int KeyState[ControllerCnt][KeyCnt];

//number of steps we are waiting before displaying next picture
int DisplayWait;

//global variables for reading
unsigned char ReadBuffer[4096]; //bytes already read but not yet processed
int ReadBufferBytes;
int ReadSizeX, ReadSizeY; //size of the frame
int ReadLeft, ReadRight, ReadTop, ReadBottom; //part of picture to be shown
int ReadOffsetX, ReadOffsetY; //free space on top and on left
int ReadPosX, ReadPosY; //position of next pixel to read
unsigned char ReadPixel[DisplaySizeY][DisplaySizeX]; //pixels of the frame
int ReadDelay;

//global variables for writing
unsigned char WriteBuffer[4096];
int WriteBufferBytes;

//start reading a new frame
void ReadStart( )
{
	ReadSizeX = -1; //size X not read yet
	ReadSizeY = -1; //size Y not read yet
	ReadPosX = 0; //read pixel 0/0 first
	ReadPosY = 0;
	memset( ReadPixel, 0, sizeof( ReadPixel ) );
	ReadDelay = -1; //delay not read yet
}

//one byte of the current frame read
void ReadByte( unsigned char Byte )
{
	int X, Y;

	//read size x
	if( ReadSizeX == -1 )
	{
		ReadSizeX = Byte;
		//calculate part of picture to show and offset
		if( ReadSizeX < DisplaySizeX )
		{
			ReadLeft = 0;
			ReadRight = ReadSizeX - 1;
			ReadOffsetX = (DisplaySizeX - ReadSizeX) / 2;
		}
		else
		{
			ReadLeft = (ReadSizeX - DisplaySizeX) / 2;
			ReadRight = ReadLeft + DisplaySizeX - 1;
			ReadOffsetX = 0;
		}
		return;
	}
	//read size y
	if( ReadSizeY == -1 )
	{
		ReadSizeY = Byte;
		//calculate part of picture to show and offset
		if( ReadSizeY < DisplaySizeY )
		{
			ReadTop = 0;
			ReadBottom = ReadSizeY - 1;
			ReadOffsetY = (DisplaySizeY - ReadSizeY) / 2;
		}
		else
		{
			ReadTop = (ReadSizeY - DisplaySizeY) / 2;
			ReadBottom = ReadTop + DisplaySizeY - 1;
			ReadOffsetY = 0;
		}
		return;
	}
	//read pixel
	if( ReadPosY < ReadSizeY )
	{
		//we are interested in this pixel
		if( ReadPosX >= ReadLeft && ReadPosX <= ReadRight && ReadPosY >= ReadTop && ReadPosY <= ReadBottom )
			//save this pixel
			ReadPixel[ReadPosY - ReadTop + ReadOffsetY][ReadPosX - ReadLeft + ReadOffsetX] = Byte;
		//go to next pixel
		ReadPosX++;
		if( ReadPosX >= ReadSizeX )
		{
			ReadPosX = 0;
			ReadPosY++;
		}
		return;
	}
	//read delay
	ReadDelay = Byte;

	//print picture on screen
	printf( "\033[%d;1f", LineNoPicture ); //go to line LineNoPicture, column 1
	for( Y = 0; Y < DisplaySizeY; Y++ )
	{
		for ( X = 0; X < DisplaySizeX; X++ )
		{
			//print pixel
			switch( ReadPixel[Y][X] >> 6 )
			{
				case 1:
					printf( "+" );
					break;
				case 2:
					printf( "*" );
					break;
				case 3:
					printf( "#" );
					break;
				default:
					printf( " " );
			}
		}
		printf( "\n" );
	}
	printf( "\033[%d;1f", LineNoEnd ); //go to line LineNoEnd, column 1
	fflush( stdout );
	//wait before displaying the next picture
	if( ReadDelay > 0 )
		DisplayWait = ((int)ReadDelay * 5000) / WaitTime; //now, we are waiting for some steps

	//start new picture
	ReadStart( );
}

//print controllers
void PrintControllers( )
{
	int Controller, Key;
	char * * Keys;
	char KeyStr[ControllerCnt][KeyCnt][64];

	//print keys
	printf( "\033[%d;1f", LineNoKeys ); //go to line LineNoKeys, column 1
	printf( "space - toggle controller keys     enter - end program\n" );	

	//get controller key-layout
	Keys = ControllerKeys[ControllerKeyIndex].Keys;
	//get key-strings (pressed / not pressed)
	for( Controller = 0; Controller < ControllerCnt; Controller++ )
		for( Key = 0; Key < KeyCnt; Key++ )
			if( KeyState[Controller][Key] )
				sprintf( KeyStr[Controller][Key], "\033[1m%c\033[0m", Keys[Controller][Key] );
			else
				sprintf( KeyStr[Controller][Key], "%c", Keys[Controller][Key] );
	//print controllers
	printf( "\033[%d;1f", LineNoControllers ); //go to line LineNoControllers, column 1
	printf( "      +---+               +---+               +---+               +---+   \n" );
	printf( "      I %s I        +---+  I %s I               I %s I        +---+  I %s I   \n", KeyStr[0][KeyUp], KeyStr[0][KeyX], KeyStr[1][KeyUp], KeyStr[1][KeyX] );
	printf( "  +---+   +---+    I %s I  +---+           +---+   +---+    I %s I  +---+   \n", KeyStr[0][KeyY], KeyStr[1][KeyY] );
	printf( "  I %s       %s I    +---+     +---+        I %s       %s I    +---+     +---+\n", KeyStr[0][KeyLeft], KeyStr[0][KeyRight], KeyStr[1][KeyLeft], KeyStr[1][KeyRight] );
	printf( "  +---+   +---+       +---+  I %s I        +---+   +---+       +---+  I %s I\n", KeyStr[0][KeyA], KeyStr[1][KeyA] );
	printf( "      I %s I           I %s I  +---+            I %s I           I %s I  +---+\n", KeyStr[0][KeyDown], KeyStr[0][KeyB], KeyStr[1][KeyDown], KeyStr[1][KeyB] );
	printf( "      +---+           +---+                   +---+           +---+       \n" );
	printf( "  select: %s     start: %s                   select: %s     start: %s         \n", KeyStr[0][KeySelect], KeyStr[0][KeyStart], KeyStr[1][KeySelect], KeyStr[1][KeyStart] );

	printf( "\033[%d;1f", LineNoEnd ); //go to line LineNoEnd, column 1
	fflush( stdout );
}

//controller key was pressed or released
void ControllerKey( int Ctr, int Key, int Pressed )
{
	//button Key of controller Ctr was pressed or released
	if( WriteBufferBytes + 4 <= sizeof( WriteBuffer ) ) //enough space in buffer
	{
		//write press and release sequence into buffer
		WriteBuffer[WriteBufferBytes + 0] = 0xFF;
		if( Pressed )
			WriteBuffer[WriteBufferBytes + 1] = 0x01;
		else
			WriteBuffer[WriteBufferBytes + 1] = 0x00;
		WriteBuffer[WriteBufferBytes + 2] = (unsigned char)Ctr;
		WriteBuffer[WriteBufferBytes + 3] = (unsigned char)Key;
		WriteBufferBytes += 4;
	}		
}

//main
int main( int ArgC, char * * ArgV )
{
	int FifoOut, FifoIn, FdMax, Loop, I, J, Ctr, Key;
	unsigned char Buffer[4096];
	struct termios StdInTermiosSave, StdInTermios;
	fd_set ReadFds, WriteFds, ExceptFds;
	struct timeval TimeOut;

	printf( "\033c" );
	printf( "bmsim: BLINKENmini simulator (version %d.%d.%d)\n", VerMaj, VerMin, VerRev );
	printf( "       done in 2002 by sphaera & 1stein\n" );
	printf( "       http://blinkenmini.1stein.no-ip.com/\n" );
	printf( "       distributed under GNU Public License - no warranty\n\n" );
	//check arguments
	if( ArgC != 3 )
	{
		printf( "bmsim <out-pipe-filename> <in-pipe-filename>\n\n" );
		return 1;
	}

	//set stdin to unbuffered and to no echo
	tcgetattr( 0, &StdInTermiosSave ); //get current settings
	memcpy( &StdInTermios, &StdInTermiosSave, sizeof( struct termios ) );
	StdInTermios.c_lflag &= ~ICANON & ~ECHO; //alter settings
	tcsetattr( 0, TCSANOW, &StdInTermios ); //set new settings

	//set start-values for processing input and output
	ReadStart( );
	WriteBufferBytes = 0;
	DisplayWait = 0;

	//set key-state to no key pressed
	for( Ctr = 0; Ctr < ControllerCnt; Ctr++ )
		for( Key = 0; Key < KeyCnt; Key++ )
			KeyState[Ctr][Key] = 0;

	//print controllers
	ControllerKeyIndex = 0;
	PrintControllers( );

	//no fifo open yet
	FifoOut = -1;
	FifoIn = -1;
	
	for( Loop = 1; Loop; )
	{
		//output-fifo not open yet
		if( FifoOut == -1 )
		{
			//open output-fifo for reading
			FifoOut = open( ArgV[1], O_RDONLY | O_NOCTTY | O_NONBLOCK );
			if( FifoOut == -1 && errno != ENXIO )
			{
				//on error
				printf( "bmsim: could not open %s: %s\n\n", ArgV[1], strerror( errno ) );
				if( FifoOut != -1 )
					close( FifoOut );
				if( FifoIn != -1 )
					close( FifoIn );
				tcsetattr( 0, TCSANOW, &StdInTermiosSave );
				return -1;
			}
		}
		//input-fifo not open yet
		if( FifoIn == -1 )
		{
			//open input-fifo for writing
			FifoIn = open( ArgV[2], O_WRONLY | O_NOCTTY | O_NONBLOCK );
			if( FifoIn == -1 && errno != ENXIO )
			{	
				//on error
				printf( "bmsim: could not open %s: %s\n\n", ArgV[2], strerror( errno ) );
				if( FifoOut != -1 )
					close( FifoOut );
				if( FifoIn != -1 )
					close( FifoIn );
				tcsetattr( 0, TCSANOW, &StdInTermiosSave );
				return -1;
			}
		}

		//check if it is possible to read from output-fifo, write to input-fifo or read from stdin
		FdMax = 0; //get maximum fd
		if( FifoOut > FdMax && FifoOut != -1 )
			FdMax = FifoOut;
		if( FifoIn > FdMax && FifoIn != -1 )
			FdMax = FifoIn;
		FD_ZERO( &ReadFds ); //check for possibility to read
		FD_SET( 0, &ReadFds ); //check stdin
		if( DisplayWait == 0 && FifoOut != -1 ) //check output-fifo if we are not waiting
			FD_SET( FifoOut, &ReadFds );
		FD_ZERO( &WriteFds ); //check for possibility to write
		if( WriteBufferBytes > 0 && FifoIn != -1 ) //check input-fifo, if we have something to write
			FD_SET( FifoIn, &WriteFds );
		FD_ZERO( &ExceptFds ); //check for exceptions
		FD_SET( 0, &ExceptFds ); //check stdin
		if( FifoOut != -1 )
			FD_SET( FifoOut, &ExceptFds ); //check output-fifo
		if( FifoIn != -1 )
			FD_SET( FifoIn, &ExceptFds ); //check input-fifo
		TimeOut.tv_usec = 1000; //set timeout very short
		TimeOut.tv_sec = 0;
		I = select( FdMax + 1, &ReadFds, &WriteFds, &ExceptFds, &TimeOut ); //do the select
		if( I == -1 )
		{
			//ignore the error "interrupted from system-call"
			if( errno != EINTR )
			{
				//error
				printf( "bmsim: could not do select: %s\n\n", strerror( errno ) );
				if( FifoOut != -1 )
					close( FifoOut );
				if( FifoIn != -1 )
					close( FifoIn );
				tcsetattr( 0, TCSANOW, &StdInTermiosSave );
				return -1;
			}
			//we were interrupted, so empty fd-sets
			FD_ZERO( &ReadFds );
			FD_ZERO( &WriteFds );
			FD_ZERO( &ExceptFds );
		}

		//exception on stdin
		if( FD_ISSET( 0, &ExceptFds ) )
		{
			//error
			printf( "bmsim: select returned exception on stdin\n\n" );
			if( FifoOut != -1 )
				close( FifoOut );
			if( FifoIn != -1 )
				close( FifoIn );
			tcsetattr( 0, TCSANOW, &StdInTermiosSave );
			return -1;
		}

		//exception on output-fifo
		if( FifoOut != -1 && FD_ISSET( FifoOut, &ExceptFds ) )
		{
			//error
			printf( "bmsim: select returned exception on output-pipe\n\n" );
			if( FifoOut != -1 )
				close( FifoOut );
			if( FifoIn != -1 )
				close( FifoIn );
			tcsetattr( 0, TCSANOW, &StdInTermiosSave );
			return -1;
		}

		//exception on input-fifo
		if( FifoIn != -1 && FD_ISSET( FifoIn, &ExceptFds ) )
		{
			//error
			printf( "bmsim: select returned exception on input-pipe\n\n" );
			if( FifoOut != -1 )
				close( FifoOut );
			if( FifoIn != -1 )
				close( FifoIn );
			tcsetattr( 0, TCSANOW, &StdInTermiosSave );
			return -1;
		}

		//stdin readable
		if( FD_ISSET( 0, &ReadFds ) )
		{
			//read from stdin
			I = read( 0, &Buffer, sizeof( Buffer ) );
			if( I == -1 )
			{
				//error
				printf( "bmsim: could not read from stdin: %s\n\n", strerror( errno ) );
				if( FifoOut != -1 )
					close( FifoOut );
				if( FifoIn != -1 )
					close( FifoIn );
				tcsetattr( 0, TCSANOW, &StdInTermiosSave );
				return -1;
			}
			//process read bytes
			for( J = 0; J < I; J++ )
			{
				//end program
				if( Buffer[J] == '\n' )
					Loop = 0;
				//next controller key layout
				else if( Buffer[J] == ' ' )
				{
					//next key layout
					ControllerKeyIndex++;
					if( ControllerKeyIndex >= count( ControllerKeys ) )
							ControllerKeyIndex = 0;
				}
				//controller key
				else
				{
					//check if this character is a controller key
					for( Ctr = 0; Ctr < ControllerCnt; Ctr++ )
					{
						for( Key = 0; Key < KeyCnt; Key++ )
						{
							if( ControllerKeys[ControllerKeyIndex].Keys[Ctr][Key] == Buffer[J] )
							{
								//set key-state to pressed
								if( KeyState[Ctr][Key] == 0 ) //key was pressed
									ControllerKey( Ctr, Key, 1 );
								KeyState[Ctr][Key] = KeyStatePressVal;
							}
						}
					}
				}
			}
		}

		//output-fifo readable
		if( FifoOut != -1 && FD_ISSET( FifoOut, &ReadFds ) )
		{
			//we are not waiting
			if( DisplayWait == 0 )
			{
				//read from the output-fifo
				I = read( FifoOut, ReadBuffer + ReadBufferBytes, sizeof( ReadBuffer ) - ReadBufferBytes );
				if( I == -1 )
				{
					//error
					printf( "bmsim: could not read from %s: %s\n\n", ArgV[1], strerror( errno ) );
					if( FifoOut != -1 )
						close( FifoOut );
					if( FifoIn != -1 )
						close( FifoIn );
					tcsetattr( 0, TCSANOW, &StdInTermiosSave );
					return -1;
				}
				//remember number of bytes in buffer
				ReadBufferBytes += I;
			}
		}

		//input-fifo writeable
		if( FifoIn != -1 && FD_ISSET( FifoIn, &WriteFds ) )
		{
			//something to write there
			if( WriteBufferBytes > 0 )
			{
				//write to input-fifo
				I = write( FifoIn, WriteBuffer, WriteBufferBytes );
				if( I == -1 )
				{
					//error
					printf( "bmsim: could not write to %s: %s\n\n", ArgV[2], strerror( errno ) );
					if( FifoOut != -1 )
						close( FifoOut );
					if( FifoIn != -1 )
						close( FifoIn );
					tcsetattr( 0, TCSANOW, &StdInTermiosSave );
					return -1;
				}
				//remove written bytes from buffer
				WriteBufferBytes -= I;
				memmove( WriteBuffer, WriteBuffer + I, WriteBufferBytes );
			}
		}

		//process read bytes while we are not waiting
		for( I = 0; DisplayWait == 0 && I < ReadBufferBytes; I++ )
			ReadByte( ReadBuffer[I] );
		//remove processed bytes from buffer
		ReadBufferBytes -= I;
		memmove( ReadBuffer, ReadBuffer + I, ReadBufferBytes );

		//decrease key-states (keys are not pressed forever)
		for( Ctr = 0; Ctr < ControllerCnt; Ctr++ )
		{
			for( Key = 0; Key < KeyCnt; Key++ )
			{
				if( KeyState[Ctr][Key] > 0 )
				{
					KeyState[Ctr][Key]--;
					if( KeyState[Ctr][Key] == 0 ) //key was released
						ControllerKey( Ctr, Key, 0 );
				}
			}
		}
		//reprint controllers
		PrintControllers( );

		//decrease display wait
		if( DisplayWait > 0 )
			DisplayWait--;
		//wait for a short time
		usleep( WaitTime );
	} //for( Loop = 1; Loop; )

	//close fifos
	if( FifoOut != -1 )
		close( FifoOut );
	if( FifoOut != -1 )
		close( FifoIn );

	//reset settings of stdin
	tcsetattr( 0, TCSANOW, &StdInTermiosSave );

	printf( "\n\n" );
	return 0;
}

